<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Ignite\Catalog\Contracts\OrderRepository as OrderRepositoryContract;
use Ignite\Catalog\Entities\CartItem;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Events\OrderEvent;
use Ignite\Catalog\Jobs\InteractWithVendor;
use Ignite\Core\Entities\User;
use Ignite\Core\Entities\Transaction;

class OrderRepository implements OrderRepositoryContract
{
    /**
     * The most basic cart query.
     *
     * @param  array $relations
     * @return \Illuminate\Database\Eloquent\Builder?
     */
    public function query($relations = [])
    {
        return Order::with(array_merge(['items'], $relations));
    }

    /**
     * Find all cart items. Not scoped by user.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function findAll()
    {
        return $this->query()->get();
    }

    /**
     * Find all cart items. Not scoped by user.
     *
     * @param  int|User $user
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function findAllByUser($user)
    {
        return $this->query()->byUser($user)->get();
    }

    /**
     * Find an order by number.
     *
     * @param  string        $number
     * @param  null|int|User $user
     * @return Order
     */
    public function find($number, $user = null)
    {
        $builder = $order = Order::with('items')->byNumber($number);

        if (! is_null($user)) {
            $builder->byUser($user);
        }

        return $builder->firstOrFail();
    }

    /**
     * Find orders by logged in user.
     *
     * @param  string $number
     * @return Order
     */
    public function findByLoggedInUser($number)
    {
        return $this->find($number, auth()->user()->getKey());
    }

    /**
     * Create a new order.
     *
     * @param  array $data
     * @param  \Illuminate\Support\Collection $items
     * @param  float $total
     * @return \Ignite\Catalog\Entities\Order
     */
    public function create($data, $items, $total)
    {
        $total = (float) str_replace(',', '', $total);

        $order = Order::create(array_merge($data, [
            'number' => 'PROCESSING',
            'notes' => '',
            'points' => $total,
            'processed' => 0,
            'cancelled' => 0,
        ]));

        /** @var CartItem $cartItem */
        foreach ($items as $cartItem) {
            $order->items()->save(new OrderItem([
                'catalog_item_id' => $cartItem->item->getKey(),
                'catalog_vendor_id' => $cartItem->item->catalog_vendor_id,
                'sku' => $cartItem->item->sku,
                'name' => $cartItem->item->name,
                'type' => $cartItem->item->type,
                'class' => $cartItem->item->class,
                'quantity' => $cartItem->quantity,
                'price' => (float) $cartItem->item->price,
                'point_value' => (float) $cartItem->item->point_value,
                'points' => (float) $cartItem->item->points,
                'created_at' => isset($data['created_at']) ? $data['created_at'] : now()
            ]));
        }

        $transaction = Transaction::create([
            'user_id' => auth()->user()->getKey(),
            'related_id' => $order->id,
            'related_name' => 'ORDER',
            'related_type' => Order::class,
            'description' => $order->number,
            'type' => 'REDEEMED',
            'value' => -$total,
            'tax_date' => now(),
            'transaction_date' => now(),
        ]);

        CartItem::where('user_id', auth()->user()->getKey())->delete();

        event(new OrderEvent($order));

        dispatch(new InteractWithVendor($order));

        return $order;
    }

    /**
     * Update the order shipping details.
     *
     * @param  string $number
     * @param  array $data
     * @return Order
     */
    public function update($number, $data)
    {
        if (! $data instanceof Collection) {
            $data = collect($data);
        }

        $order = $this->find($number);
        $order->update($data->only([
            'ship_name', 'ship_email', 'ship_phone', 'ship_address_1', 'ship_address_2',
            'ship_address_3', 'ship_city', 'ship_state', 'ship_postal', 'ship_country'
        ])->toArray());

        return $order;
    }

    /**
     * Cancel the order.
     *
     * @param  string $number
     * @return Order
     */
    public function cancel($number)
    {
        $order = $this->find($number);
        $order->cancelled = 1;
        $order->cancelled_at = now();

        if (! $order->relationLoaded('items')) {
            $order->load('items');
        }

        $totalPointsToReturn = 0;

        foreach ($order->items as $item) {
            if (! $item->isCancelled()) {
                $totalPointsToReturn += ($item->points * $item->quantity);
                $item->cancel([
                    'quantity' => $item->quantity
                ])->save();
            }
        }

        $order->save();

        if ($totalPointsToReturn > 0) {
            Transaction::create([
                'user_id' => $order->user_id,
                'related_id' => $order->id,
                'related_name' => 'ORDER',
                'related_type' => Order::class,
                'description' => $order->number,
                'type' => 'CANCELLED',
                'value' => $totalPointsToReturn,
                'tax_date' => now(),
                'transaction_date' => now(),
            ]);
        }

        return $order;
    }

    /**
     * Process the order.
     *
     * @param  string $number
     * @return Order
     */
    public function process($number)
    {
        $order = $this->find($number);
        $order->processed = 1;
        $order->processed_at = now();

        foreach ($order->items as $item) {
            $item->process([
                'quantity' => $item->quantity
            ])->save();
        }

        $order->save();

        return $order;
    }

    /**
     * Delete the records with the given IDs.
     *
     * @param  string|array $ids
     * @return bool
     */
    public function delete($ids)
    {
        DB::beginTransaction();

        try {
            $records = Order::byIds($ids)->get();

            /** @var Order $record */
            foreach ($records as $record) {
                $this->cancel($record->number);
                $record->items()->delete();
                $record->delete();
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new \DomainException("Unable to delete records with IDs: {$ids} - {$e->getMessage()}");
        }

        DB::commit();

        return true;
    }
}
