<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Catalog\Contracts\CatalogItemRepository;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Http\Forms\ItemForm;
use Ignite\Catalog\Models\Grid\ItemsTable;
use Ignite\Core\Http\Controllers\Controller;
use Kris\LaravelFormBuilder\FormBuilder;

class ItemController extends Controller
{
    /**
     * Display all items.
     *
     * @param  ItemsTable $table
     * @param  Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(ItemsTable $table, Request $request)
    {
        if ($request->has('visibility')) {
            $table->with('visibility', $request->input('visibility', '-1'));
        }

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.items.index');
    }

    /**
     * Show a single item.
     *
     * @param  Item $item
     * @return \Illuminate\Http\Response
     */
    public function show(Item $item)
    {
        return view('Catalog::admin.items.show', compact('item'));
    }

    /**
     * Create a new catalog item.
     *
     * @param  Request $request
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\View\View
     */
    public function create(Request $request, FormBuilder $formBuilder)
    {
        $type = $request->get('type', '');

        if (empty($type)) {
            return view('Catalog::admin.items.type');
        }

        if (! in_array($type, ['simple', 'configurable'])) {
            $this->flashError('Catalog::item.misc.create.unknown_type', compact('type'));
            return back();
        }

        $form = $formBuilder->create(ItemForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.items.create')
        ], compact('type'));

        return view('Catalog::admin.items.create', compact('type', 'form'));
    }

    /**
     * Store the item.
     *
     * @param  Request $request
     * @param  FormBuilder $formBuilder
     * @param  CatalogItemRepository $catalogItemRepository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request, FormBuilder $formBuilder, CatalogItemRepository $catalogItemRepository)
    {
        $form = $formBuilder->create(ItemForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $data = array_merge($request->except('_token', 'back'), $form->getFieldValues(true));
            $item = $catalogItemRepository->create($data);
            $this->flashSuccess('Catalog::item.store.success', ['name' => $item->name]);
            if ($request->has('back')) {
                return redirect()->route('admin.catalog.items.index');
            }
            return redirect()->route('admin.catalog.items.edit', $item);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::item.store.failure');
            return redirect()->route('admin.catalog.items.index');
        }
    }

    /**
     * Update an existing catalog item.
     *
     * @param  Item $item
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\View\View
     */
    public function edit(Item $item, FormBuilder $formBuilder)
    {
        $type = $item->type;

        $form = $formBuilder->create(ItemForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.items.update', compact('item')),
            'model' => $item
        ], compact('type'));

        return view('Catalog::admin.items.edit', compact('type', 'form', 'item'));
    }

    /**
     * Update the item.
     *
     * @param  string $item
     * @param  Request $request
     * @param  FormBuilder $formBuilder
     * @param  CatalogItemRepository $catalogItemRepository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update($item, Request $request, FormBuilder $formBuilder, CatalogItemRepository $catalogItemRepository)
    {
        $form = $formBuilder->create(ItemForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }
        try {
            $data = array_merge($request->except('_token', 'back'), $form->getFieldValues(true));
            $item = $catalogItemRepository->update($item, $data);
            $this->flashSuccess('Catalog::item.update.success', ['name' => $item->name]);
            if ($request->has('back')) {
                return redirect()->route('admin.catalog.items.index');
            }
            return redirect()->route('admin.catalog.items.edit', ['item' => $item]);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashSuccess('Catalog::item.update.failure');
            return redirect()->route('admin.catalog.items.index');
        }
    }

    /**
     * Duplicate an item, only if its a simple item.
     *
     * @param  Item $item
     * @return \Illuminate\Http\RedirectResponse
     */
    public function duplicate(Item $item)
    {
        try {
            if ($item->type !== 'simple') {
                throw new \InvalidArgumentException('Only simple items can be duplicated.');
            }
            $id = uniqid();

            $copy = $item->replicate();
            $copy->name = "{$item->name} (Copy $id)";
            $copy->sku  = "{$item->sku}_$id";
            $copy->code = "{$item->code}-$id";
            $copy->save();

            flash()->success("Item `{$item->name}` was duplicated successfully.")->important();
        } catch (\InvalidArgumentException $e) {
            flash()->error($e->getMessage())->important();
        } catch (\Exception $e) {
            $this->logException($e);
            flash()->error("Unable to duplicate item `{$item->name}`.")->important();
        }

        return redirect()->route('admin.catalog.items.edit', $copy);
    }

    /**
     * Toggle an item field value.
     *
     * @param  Item $item
     * @param  string $field
     * @param  string $value
     * @return \Illuminate\Http\RedirectResponse
     */
    public function toggle(Item $item, $field, $value)
    {
        $value = (int) $value;

        switch ($field) {
            case 'active':
                $action = $value === 1 ? 'was activated' : 'was deactivated';
                break;
            case 'vendor_active':
                $action = $value === 1 ? 'vendor was enabled' : 'vendor was disabled';
                break;
            case 'visibility':
                $action = $value === 1 ? 'was shown in catalog' : 'was hidden in catalog';
                break;
            default:
                $action = 'was not updated';
        }

        if (in_array($field, ['active', 'vendor_active', 'visibility'])) {
            $item->setAttribute($field, $value);
            $item->save();
        }

        flash()->success("{$item->name} {$action}.")->important();

        return redirect()->route('admin.catalog.items.index');
    }

    /**
     * Delete multiple records at once.
     *
     * @param  Request $request
     * @param  CatalogItemRepository $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(Request $request, CatalogItemRepository $repository)
    {
        $ids = $request->get('ids');

        try {
            $repository->delete($ids);
            $message = "The following item IDs were deleted successfully: {$ids}.";
            $level = 'success';
        } catch (\Exception $e) {
            $message = $e->getMessage();
            $level = 'error';
        }

        flash()->message($message, $level)->important();

        return redirect()->back();
    }
}
