<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Illuminate\Bus\Queueable;
use Illuminate\Http\Response as HttpResponse;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Ignite\Vendor\Hawk\ClosedLoopIndividualOrderLine;
use Ignite\Vendor\Hawk\PaymentType;
use Ignite\Vendor\Hawk\RecipientAddress;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Vendor\Hawk\Client as HawkApi;

class Giftcard extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Execute the job.
     *
     * @param  HawkApi $hawkApi
     * @return bool
     */
    public function handle(HawkApi $hawkApi)
    {
        $requestId = $this->buildRequestId();

        logger('Request data', $this->buildClosedLoopIndividualRequest()->toArray());

        try {
            $response = $hawkApi->submitClosedLoopIndividual($this->buildClosedLoopIndividualRequest(), $requestId);

            if ($response instanceof SubmitClosedLoopIndividualResponse) {
                if ($response->getStatusCode() === HttpResponse::HTTP_OK) {
                    //logger('200 - Order complete', ['body' => (string) $response->getBody()->getContents()]);
                    return $this->complete($response, $requestId);
                } else {
                    // OrderInfo request? Submit a subsequent orderInfo request 10 seconds from now to see if the
                    // order has completed, if not let a cron job do it 5 minutes from now?
                    // logger('202 - Order not complete', ['body' => (string) $response->getBody()->getContents()]);
                    return $this->complete($response, $requestId);
                }
            }

            if ($response instanceof ErrorResponse) {
                return $this->error($response, $requestId);
            }
        } catch (\Exception $e) {
            $response = new ErrorResponse(['exception' => [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode(),
                'trace' => $e->getTraceAsString()
            ]]);

            return $this->error($response, $requestId);
        }

        return false;
    }

    /**
     * Build the request for the giftcard Individual.
     *
     * @return SubmitClosedLoopIndividualRequest
     */
    private function buildClosedLoopIndividualRequest()
    {
        return SubmitClosedLoopIndividualRequest::fromAssociativeArray([
            'paymentType' => PaymentType::ACH_DEBIT,
            'poNumber' => $this->order->number,
            'clientProgramNumber' => $this->getProductId(),
            'orderDetails' => $this->buildIndividualOrderLines(),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildIndividualOrderLines()
    {
        return array_map(function () {
            return $this->buildIndividualOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return ClosedLoopIndividualOrderLine
     */
    protected function buildIndividualOrderLine()
    {
        if (empty($this->orderItem->item->vendor_meta['content_provider_code'])) {
            throw new \DomainException("Missing content provider code for Item ID: {$this->orderItem->item->id}");
        }

        return ClosedLoopIndividualOrderLine::fromAssociativeArray([
            'recipient' => $this->getRecipientData(),
            'amount' => (int) $this->orderItem->points,
            'quantity' => $this->orderItem->quantity,
            'contentProvider' => $this->orderItem->item->vendor_meta['content_provider_code'],
            'groupingId' => $this->order->number
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return array
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return [
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'id' => $this->order->user_id,
            'phoneNumber' => $this->order->ship_phone,
            'address' => RecipientAddress::fromAssociativeArray([
                'line1' => $this->order->ship_address_1,
                'line2' => $this->order->ship_address_2,
                'line3' => $this->order->ship_address_3,
                'city' => $this->order->ship_city,
                'region' => $this->order->ship_state,
                'postalCode' => $this->order->ship_postal,
                'country' => $this->formatCountry($this->order->ship_country),
            ])
        ];
    }

    /**
     * Format country for a RecipientAddress model.
     *
     * @param  string $country
     * @return string
     */
    protected function formatCountry($country)
    {
        if ($country === 'US') {
            return 'USA';
        }

        if ($country === 'CA') {
            return 'CAN';
        }

        throw new \DomainException("Unsupported country: $country. Blackhawk only support USA and Canada.");
    }

    /**
     * Get the item quantity and cast to an integer for comparison.
     *
     * @return int
     */
    protected function getOrderItemQuantity()
    {
        return (int) $this->orderItem->getItemQuantity();
    }
}
