<?php

namespace Ignite\Catalog\Entities;

use Ignite\Catalog\Presenters\ItemPresenter;
use Ignite\Core\Facades\Format;
use Ignite\Packages\Presenter\Traits\Presentable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Ignite\Catalog\Contracts\CardType;
use Ignite\Core\Traits\Markdownable;

/**
 * Class Item
 *
 * @property int $id
 * @property int $catalog_vendor_id
 * @property int $catalog_id
 * @property string $code
 * @property string $sku
 * @property string $type
 * @property string $class
 * @property string $name
 * @property string $short_description
 * @property string $description
 * @property string $terms
 * @property string $manufacturer
 * @property string $image
 * @property float $msrp
 * @property float $cost
 * @property float $price
 * @property float $price_markup
 * @property float $price_margin
 * @property float $point_value
 * @property float $points
 * @property float $points_min
 * @property float $points_max
 * @property int $visibility
 * @property int $active
 * @property int $vendor_active
 * @property array $vendor_meta
 * @property \Carbon\Carbon $created_at
 * @property \Carbon\Carbon $updated_at
 * @property Vendor $vendor
 * @property Catalog $catalog
 * @method ItemPresenter present()
 */
class Item extends Model
{
    use Markdownable;
    use Presentable;

    /** @var string */
    protected $presenter = \Ignite\Catalog\Presenters\ItemPresenter::class;

    /** @var string */
    protected $primaryKey = 'id';

    /** @var string */
    protected $table = 'catalog_item';

    /** @var array */
    protected $guarded = [];

    /** @var array */
    protected $casts = [
        'vendor_meta' => 'json'
    ];

    /**
     * Boot the eloquent listeners.
     */
    public static function boot()
    {
        parent::boot();

        static::saving(function ($model) {
            if (empty($model->sku)) {
                $model->sku = str_slug(strtolower($model->name), '_');
            }
        });
    }

    /**
     * Scope an item query by id.
     *
     * @param Builder $query
     * @param $id
     */
    public function scopeById(Builder $query, $id)
    {
        $query->where('id', $id);
    }

    /**
     * Scope an item query by id.
     *
     * @param Builder $query
     * @param array $ids
     */
    public function scopeByIds(Builder $query, $ids)
    {
        if (is_string($ids)) {
            $ids = explode(',', $ids);
        }

        $query->whereIn(static::getKeyName(), $ids);
    }

    /**
     * Scope an item query by sku.
     *
     * @param Builder $query
     * @param $sku
     */
    public function scopeBySku(Builder $query, $sku)
    {
        $query->where('id', $sku);
    }

    /**
     * Scope an item query by code.
     *
     * @param Builder $query
     * @param $code
     */
    public function scopeByCode(Builder $query, $code)
    {
        $query->where('id', $code);
    }

    /**
     * Scope an item query by catalog.
     *
     * @param Builder $query
     * @param Catalog $catalog
     */
    public function scopeByCatalog(Builder $query, $catalog)
    {
        if ($catalog instanceof Catalog) {
            $catalog = $catalog->getKey();
        }

        $query->where('catalog_id', $catalog);
    }

    /**
     * Scope an item query by category.
     *
     * @param Builder $query
     * @param string  $category
     */
    public function scopeByCategory(Builder $query, $category)
    {
        if ($category instanceof Category) {
            $category = $category->getKey();
        }

        $query->whereHas('categories', function ($builder) use ($category) {
            $builder->where('category_id', $category);
        });
    }

    /**
     * Scope an item query by catalog.
     *
     * @param Builder $query
     */
    public function scopeOnlyVisible(Builder $query)
    {
        $query->where('visibility', 1);
    }

    /**
     * Match the query against a keyword.
     *
     * @param Builder $query
     * @param string  $keyword
     */
    public function scopeMatchAgainst(Builder $query, $keyword)
    {
        $fields = [
            3 => 'manufacturer',
            2 => 'name',
            1 => 'description'
        ];

        $columns = implode(', ', array_values($fields));
        $keyword = $this->fullTextWildcards($keyword);

        foreach ($fields as $weight => $column) {
            $query->selectRaw("MATCH ({$column}) AGAINST (? IN BOOLEAN MODE) AS relevance_score_$column", [$keyword]);
        }

        $query->whereRaw("MATCH ({$columns}) AGAINST (? IN BOOLEAN MODE)", $keyword);

        $orderBy = '';
        foreach ($fields as $weight => $column) {
            $orderBy .= "(relevance_score_$column * $weight) + ";
        }

        $orderBy = rtrim($orderBy, " + ") . ' desc';

        $query->orderByRaw($orderBy);
    }

    /**
     * Scope a query by name in ascending order.
     *
     * @param Builder $query
     */
    public function scopeOrderByNameAsc(Builder $query)
    {
        $query->orderBy('name', 'asc');
    }

    /**
     * Scope a query by name in descending order.
     *
     * @param Builder $query
     */
    public function scopeOrderByNameDesc(Builder $query)
    {
        $query->orderBy('name', 'desc');
    }

    /**
     * Replaces spaces with full text search wildcards
     *
     * @param string $term
     * @return string
     */
    protected function fullTextWildcards($term)
    {
        // remove symbols used by MySQL
        $symbols = ['-', '+', '<', '>', '@', '(', ')', '~', '"'];
        $term = str_replace($symbols, '', $term);
        $words = explode(' ', $term);

        foreach ($words as $key => $word) {
            // applying + operator (required word) only big words
            // because smaller ones are not indexed by mysql
            if (strlen($word) >= 3) {
                $words[$key] = '+' . $word . '*';
            }
        }

        return implode(' ', $words);
    }

    /**
     * The relationship to the catalog.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function catalog()
    {
        return $this->belongsTo(Catalog::class, 'catalog_id', 'id');
    }

    /**
     * The relationship to the vendor.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function vendor()
    {
        return $this->belongsTo(\Ignite\Catalog\Entities\Vendor::class, 'catalog_vendor_id', 'id');
    }

    /**
     * The related categories for this item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function categories()
    {
        return $this->belongsToMany(\Ignite\Catalog\Entities\Category::class, 'catalog_category_item', 'item_id', 'category_id');
    }

    /**
     * The relationship to the attributes associated with this item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function attributes()
    {
        return $this->belongsTo(\Ignite\Catalog\Entities\AttributeItem::class, 'item_id', 'id')->orderBy('position', 'asc');
    }

    /**
     * The product options.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function options()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\Option::class, 'super_id', 'id')->orderBy('position', 'asc');
    }

    /**
     * The associated product options.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function associated()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\Option::class, 'item_id', 'id')->orderBy('position', 'asc');
    }

    /**
     * The relationship to the item in the favorites table. If one exists.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function favorite()
    {
        return $this->hasOne(\Ignite\Catalog\Entities\Favorite::class, 'item_id', 'id')
                    ->where('user_id', auth()->user()->getKey());
    }

    /**
     * The relationship to the item in the favorites table. If one exists.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasMany
     */
    public function favorites()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\Favorite::class, 'item_id', 'id');
    }

    /**
     * Get an instance of the item type object.
     *
     * @return mixed
     */
    public function getTypeInstance()
    {
        $map = [
            'simple' => \Ignite\Catalog\Models\Product\Types\Simple::class,
            'configurable' => \Ignite\Catalog\Models\Product\Types\Configurable::class,
        ];

        return app()->make($map[$this->type], ['item' => $this]);
    }

    /**
     * Get an instance of the item classification object.
     *
     * @return mixed
     */
    public function getClassInstance()
    {
        $map = [
            'merchandise' => \Ignite\Catalog\Models\Product\Classifications\Merchandise::class,
            'giftcard' => \Ignite\Catalog\Models\Product\Classifications\Card::class,
            'egift' => \Ignite\Catalog\Models\Product\Classifications\Egift::class,
            'prepaid' => \Ignite\Catalog\Models\Product\Classifications\Prepaid::class,
            'reloadable' => \Ignite\Catalog\Models\Product\Classifications\Reloadable::class,
        ];

        return app()->make($map[$this->class], ['item' => $this]);
    }

    /**
     * Determine if the item is a gift card.
     *
     * @return bool
     */
    public function isCardType()
    {
        return $this->getClassInstance() instanceof CardType;
    }

    /**
     * Determine if the item is merchandise.
     *
     * @return bool
     */
    public function isMerchandise()
    {
        return $this->getClassInstance() instanceof \Ignite\Catalog\Models\Product\Classifications\Merchandise;
    }

    /**
     * Determine if the item is a gift card.
     *
     * @return bool
     */
    public function isGiftcard()
    {
        return $this->getClassInstance() instanceof \Ignite\Catalog\Models\Product\Classifications\Card;
    }

    /**
     * Determine if the item is an eGift card.
     *
     * @return bool
     */
    public function isEgift()
    {
        return $this->getClassInstance() instanceof \Ignite\Catalog\Models\Product\Classifications\Egift;
    }

    /**
     * Determine if the item is an prepaid non-reloadable visa card.
     *
     * @return bool
     */
    public function isPrepaid()
    {
        return $this->getClassInstance() instanceof \Ignite\Catalog\Models\Product\Classifications\Prepaid;
    }

    /**
     * Determine if the item is an prepaid reloadable visa card.
     *
     * @return bool
     */
    public function isReloadable()
    {
        return $this->getClassInstance() instanceof \Ignite\Catalog\Models\Product\Classifications\Reloadable;
    }

    /**
     * Determine whether the item is configurable.
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return $this->type === 'configurable';
    }

    /**
     * Determine whether the item is simple.
     *
     * @return bool
     */
    public function isSimple()
    {
        return $this->type === 'simple';
    }

    /**
     * Is the item visible.
     *
     * @return bool
     */
    public function isVisible()
    {
        return $this->visibility === 1;
    }

    /**
     * Is the item visible.
     *
     * @return bool
     */
    public function isInvisible()
    {
        return $this->visibility === 0;
    }

    /**
     * Is the item active.
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->active === 1;
    }

    /**
     * Is the item inactive.
     *
     * @return bool
     */
    public function isInactive()
    {
        return $this->active === 0;
    }

    /**
     * Is the vendor active.
     *
     * @return bool
     */
    public function isVendorActive()
    {
        return $this->vendor_active === 1;
    }

    /**
     * Is the vendor inactive.
     *
     * @return bool
     */
    public function isVendorInactive()
    {
        return $this->vendor_active === 0;
    }

    /**
     * The image url on the correct storage location.
     *
     * @return string
     */
    public function getImageUrl()
    {
        if (! $this->image) {
            return '';
        }

        $path = $this->vendor->image_path . '/' . $this->image;

        // @todo Refactor so the call to Storage is hidden behind a dependency that we own
        if (Storage::disk('s3')->has($path)) {
            return Storage::disk('s3')->url($path);
        }

        return Storage::disk('ignite3')->url($path);
    }

    /**
     * The image path.
     *
     * @return string
     */
    public function getImagePath()
    {
        return $this->vendor->image_path . '/';
    }

    /**
     * Attach an image to the record and upload to storage.
     *
     * @param  UploadedFile $image
     * @return $this
     */
    public function attachImage(UploadedFile $image)
    {
        $path = $this->getImagePath();
        $name = $this->code . '.' . $image->getClientOriginalExtension();
        $disk = config('catalog.disk');

        $image->storePubliclyAs($path, $name, $disk);

        $this->image = $name;

        return $this;
    }

    /**
     * Attach multiple options to the current item.
     *
     * @param  array $options
     * @return $this
     */
    public function attachOptions(array $options)
    {
        $options = collect($options)->flatten(1);

        $this->points_min = $options->min('value');
        $this->points_max = $options->max('value');

        foreach ($options as $index => $data) {
            if (empty($data['id']) || strtolower($data['id']) === 'new') {
                unset($data['id']);
            }

            $this->options()->newModelInstance()->newQuery()->updateOrCreate([
                'super_id' => $this->getKey(),
                'item_id' => $data['item_id'],
                'attribute_id' => $data['attribute_id'],
            ], $data);
        }

        $this->save();

        return $this;
    }

    /**
     * Attach categories to the item.
     *
     * @param  array $categories
     * @return $this
     */
    public function attachCategories(array $categories)
    {
        $categories = collect($categories)->keyBy(function ($category) {
            return $category;
        })->map(function ($category) {
            return [
                'position' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        })->toArray();


        $this->categories()->sync($categories);

        return $this;
    }

    /**
     * The number of points required to redeem the item, formatted.
     *
     * @return string
     */
    public function getPoints()
    {
        return Format::amount($this->points);
    }

    /**
     * The canonical item url.
     *
     * @param  array $append
     * @return string
     */
    public function getUrl($append = [])
    {
        return route('catalog.item.show', array_merge($append, ['item' => $this->getKey()]));
    }

    /**
     * The url to favorite this item.
     *
     * @return string
     */
    public function getFavoriteUrl()
    {
        if (is_null($this->favorite)) {
            return route('catalog.favorites.store', ['item' => $this->getKey()]);
        }

        return route('catalog.favorites.destroy', ['item' => $this->getKey()]);
    }

    /**
     * Determine if the item is a favorite of the current user.
     *
     * @return bool
     */
    public function isFavorited()
    {
        return (! is_null($this->favorite));
    }

    /**
     * The description excerpt.
     *
     * @param  int $limit
     * @return string
     */
    public function getExcerpt($limit = 25)
    {
        return str_replace("\n", "<br />\n", substr($this->description, 0, $limit)) . ((strlen($this->description) > $limit) ? '...' : '');
    }

    /**
     * Set the sku attribute and automatically set the code.
     *
     * @param string $sku
     */
    public function setSkuAttribute($sku)
    {
        $this->attributes['sku'] = str_slug(strtolower($sku), '_');
        $this->attributes['code'] = str_slug(strtolower($sku), '-');
    }

    /**
     * When the markup is greater than 1, we divide by 100.
     *
     * @param string $markup
     */
    public function setPriceMarkupAttribute($markup)
    {
        $markup = (float) $markup;

        if ($markup > 1) {
            $markup = $markup / 100;
        }

        $this->attributes['price_markup'] = number_format($markup, 4);
    }

    /**
     * Format the price markup.
     *
     * @return float|int
     */
    public function getPriceMarkupAttribute()
    {
        return $this->attributes['price_markup'] * 100;
    }

    /**
     * When the margin is greater than 1, we divide by 100.
     *
     * @param string $margin
     */
    public function setPriceMarginAttribute($margin)
    {
        $margin = (float) $margin;

        if ($margin > 1) {
            $margin = $margin / 100;
        }

        $this->attributes['price_margin'] = number_format($margin, 4);
    }

    /**
     * Format the price margin.
     *
     * @return float|int
     */
    public function getPriceMarginAttribute()
    {
        return $this->attributes['price_margin'] * 100;
    }

    /**
     * Set the active column.
     *
     * @param mixed $active
     */
    public function setActiveAttribute($active)
    {
        $this->attributes['active'] = (int) $active;
    }

    /**
     * Set the vendor active column.
     *
     * @param mixed $vendorActive
     */
    public function setVendorActiveAttribute($vendorActive)
    {
        $this->attributes['vendor_active'] = (int) $vendorActive;
    }

    /**
     * Set the visibility column.
     *
     * @param mixed $visibility
     */
    public function setVisibilityAttribute($visibility)
    {
        $this->attributes['visibility'] = (int) $visibility;
    }

    /**
     * Format the description.
     *
     * @param  bool $plain
     * @return string
     */
    public function getDescription($plain = false)
    {
        $description = $this->description;

        return ($plain) ? $description : $this->markdown($description);
    }

    /**
     * Format the terms.
     *
     * @param  bool $plain
     * @return string
     */
    public function getTerms($plain = false)
    {
        $terms = $this->terms;

        return ($plain) ? $terms : $this->markdown($terms);
    }
}
