<?php

namespace Ignite\Catalog\Models\Product;

use DomainException;
use Illuminate\Http\Request;
use Ignite\Catalog\Entities\Item;

class BuyRequest
{
    const PARAM_ITEM_ID = 'item_id';
    const PARAM_QUANTITY = 'quantity';
    const PARAM_OPTIONS = 'options';

    /**
     * @var int
     */
    private $itemId;

    /**
     * @var int
     */
    private $quantity;

    /**
     * @var array
     */
    private $options;

    /**
     * @var Item
     */
    private $item;

    /**
     * Create an instance from a HTTP request.
     *
     * @param  Request $request
     * @return static
     */
    public static function fromHttpRequest(Request $request)
    {
        if (! $request->has(static::PARAM_ITEM_ID)) {
            throw new DomainException('Unable to find the item ID.');
        }

        if (! $request->has(static::PARAM_QUANTITY)) {
            throw new DomainException('Unable to find the quantity.');
        }

        $options = [];
        if ($request->has(static::PARAM_OPTIONS)) {
            $options = $request->get(static::PARAM_OPTIONS, []);
        }

        return new static(
            (int) $request->get(static::PARAM_ITEM_ID),
            (int) $request->get(static::PARAM_QUANTITY),
            $options
        );
    }

    /**
     * BuyRequest constructor.
     *
     * @param int   $itemId
     * @param int   $quantity
     * @param array $options
     */
    public function __construct($itemId, $quantity, $options = [])
    {
        $this->itemId = $itemId;
        $this->quantity = $quantity;
        $this->options = $options;
        $this->item = Item::findOrFail($this->itemId);

        if ($this->item->isConfigurable() && empty($options)) {
            throw new DomainException('You can only assign options to configurable items.');
        }

        if ($this->item->isConfigurable()) {
            $this->item = $this->swapWithAssociatedItem();
        }
    }

    /**
     * The item being checked out.
     *
     * @return Item
     */
    public function getItem()
    {
        return $this->item;
    }

    /**
     * The quantity number of the item.
     *
     * @return int
     */
    public function getQuantity()
    {
        return (int) $this->quantity;
    }

    /**
     * The number of points for the item.
     *
     * @return int
     */
    public function getPoints()
    {
        return (float) $this->item->points;
    }

    /**
     * Prepare the data for storage.
     *
     * @return array
     */
    public function toArray()
    {
        return [
            'user_id' => auth()->user()->getKey(),
            'catalog_id' => $this->item->catalog_id,
            'item_id' => $this->item->getKey(),
            'super_id' => $this->itemId,
            'catalog_vendor_id' => $this->item->catalog_vendor_id,
            'quantity' => $this->getQuantity(),
            'points' => $this->getPoints(),
            'price' => $this->item->price
        ];
    }

    /**
     * Get the first item from the options to use as the lookup ID.
     *
     * @return Item
     */
    private function swapWithAssociatedItem()
    {
        return Item::findOrFail(collect($this->options)->first());
    }
}
