<?php

namespace Ignite\Catalog\Tests\Unit\Models\Hawk;

use Ignite\Catalog\Models\Hawk\RecipientAddress;
use Ignite\Tests\TestCase;

class RecipientAddressTest extends TestCase
{
    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_can_create_an_instance_from_an_associative_array()
    {
        $this->assertInstanceOf(RecipientAddress::class, RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => '75062',
            'country' => 'USA',
        ]));
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_line1_exceeds_character_limit()
    {
        RecipientAddress::fromAssociativeArray([
            'line1' => str_repeat('X', 36),
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => '75062',
            'country' => 'USA',
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_city_exceeds_character_limit()
    {
        RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'city' => str_repeat('X', 36),
            'region' => 'TX',
            'postalCode' => '75062',
            'country' => 'USA',
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_region_exceeds_character_limit()
    {
        RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'city' => 'Irving',
            'region' => str_repeat('X', 36),
            'postalCode' => '75062',
            'country' => 'USA',
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_postal_exceeds_character_limit()
    {
        RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => str_repeat('X', 11),
            'country' => 'USA',
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_country_exceeds_character_limit()
    {
        RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => '75062',
            'country' => "CANADA",
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_country_is_not_supported()
    {
        RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => '75062',
            'country' => "IRL",
        ]);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_will_include_addresslines_and_postalcodeext_when_available()
    {
        $recipientAddress = RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'line2' => 'Suite 150',
            'line3' => 'Office 13',
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => '75062',
            'postalCodeExt' => '1123',
            'country' => "USA",
        ]);

        $this->assertArrayHasKey('line2', $recipientAddress->toArray());
        $this->assertArrayHasKey('line3', $recipientAddress->toArray());
        $this->assertArrayHasKey('postalCodeExt', $recipientAddress->toArray());
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_will_not_include_addresslines_and_postalcodeext_when_not_available()
    {
        $recipientAddress = RecipientAddress::fromAssociativeArray([
            'line1' => '102 Decker Ct',
            'line2' => null,
            'city' => 'Irving',
            'region' => 'TX',
            'postalCode' => '75062',
            'postalCodeExt' => '',
            'country' => "USA",
        ]);

        $this->assertArrayNotHasKey('line2', $recipientAddress->toArray());
        $this->assertArrayNotHasKey('line3', $recipientAddress->toArray());
        $this->assertArrayNotHasKey('postalCodeExt', $recipientAddress->toArray());
    }
}
