<?php

namespace Ignite\Catalog\Entities;

use Illuminate\Database\Eloquent\Model;

class Attribute extends Model
{
    /** @var string */
    protected $table = 'catalog_attribute';

    /** @var string */
    protected $primaryKey = 'id';

    /** @var array */
    protected $guarded = [];

    /**
     * The relationship to the associated items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function items()
    {
        return $this->belongsToMany(\Ignite\Catalog\Entities\Item::class, 'catalog_attribute_item', 'item_id', 'attribute_id')
                    ->orderBy('position', 'asc');
    }

    /**
     * The relationship to the super/parent item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function super()
    {
        return $this->belongsToMany(\Ignite\Catalog\Entities\Item::class, 'catalog_attribute_item', 'super_id', 'attribute_id')
                    ->orderBy('position', 'asc');
    }

    /**
     * The options related to this attribute for the given item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function options()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\Option::class, 'attribute_id', 'item_id')
                    ->orderBy('position', 'asc');
    }

    /**
     * Get options for the given item.
     *
     * @param  Item $item
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getOptionsForItem(Item $item)
    {
        return $this->options()->where('item_id', $item->getKey())->get();
    }

    /**
     * Set the use_in_filter column value reliably.
     *
     * @param $value
     */
    public function setUseInFilterAttribute($value)
    {
        $this->attributes['use_in_filter'] = (int) $value;
    }

    /**
     * The array of presentable data for the view.
     *
     * @return array
     */
    public function toPresentableArray()
    {
        return [
            'code' => $this->code,
            'type' => $this->type,
            'use_in_filter' => $this->use_in_filter ? 'Yes' : 'No'
        ];
    }
}
