<?php

namespace Ignite\Catalog\Services;

use GuzzleHttp\Client;
use Ignite\Catalog\Models\Hawk\Request\Request;
use Ignite\Catalog\Models\Hawk\Request\SubmitClosedLoopBulkRequest;
use Ignite\Catalog\Models\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Catalog\Models\Hawk\Request\SubmitClosedLoopAnonymousRequest;
use Ignite\Catalog\Models\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Catalog\Models\Hawk\Response\ErrorResponse;
use Ignite\Catalog\Models\Hawk\Response\SubmitClosedLoopBulkResponse;
use Ignite\Catalog\Models\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Catalog\Models\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Catalog\Models\Hawk\Response\SubmitClosedLoopAnonymousResponse;
use Ignite\Catalog\Models\Hawk\Ssl;
use Ignite\Catalog\Vendors\ResponseException;
use Ignite\Catalog\Vendors\TracksRequestAttempts;

class HawkApi
{
    use TracksRequestAttempts;

    /**
     * @var string
     */
    protected $merchantId;

    /**
     * @var Ssl
     */
    protected $ssl;

    /**
     * @var Client
     */
    protected $client;

    /**
     * The production endpoint.
     * @var string
     */
    protected $liveEndpoint = 'https://api.blackhawknetwork.com/rewardsOrderProcessing/v1/';

    /**
     * The testing endpoint.
     * @var string
     */
    protected $testEndpoint = 'https://apipp.blackhawknetwork.com/rewardsOrderProcessing/v1/';

    /**
     * Flag to indicate if we are in test mode.
     * @var bool
     */
    protected $testMode;

    /**
     * HawkApi constructor.
     *
     * @param string      $merchantId
     * @param Ssl         $ssl
     * @param Client|null $client
     * @param bool        $testMode
     */
    public function __construct($merchantId, Ssl $ssl, Client $client = null, $testMode = false)
    {
        $this->merchantId = $merchantId;
        $this->ssl = $ssl;
        $this->client = is_null($client) ? $this->getDefaultClient() : $client;
        $this->testMode = $testMode;
    }

    /**
     * Submit an giftcard individual order.
     *
     * @param  SubmitEGiftIndividualRequest $request
     * @param  string                       $requestId
     * @param  int                          $millisecondsToWait
     * @return SubmitClosedLoopAnonymousResponse|ErrorResponse
     * @throws ResponseException
     */
    public function submitEgiftIndividual(SubmitEGiftIndividualRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $response = $this->buildPostRequest('submitEgiftIndividual', $request, $requestId, $millisecondsToWait);

        if (in_array($response->getStatusCode(), [400, 404])) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitEGiftIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop individual order. Giftcard.
     *
     * @param  SubmitClosedLoopIndividualRequest $request
     * @param  null $requestId
     * @param  int $millisecondsToWait
     * @return ErrorResponse|SubmitClosedLoopAnonymousResponse|static
     * @throws ResponseException
     */
    public function submitClosedLoopIndividual(SubmitClosedLoopIndividualRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitClosedLoopIndividual', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop bulk order. Giftcard, multiple quantities.
     *
     * @param  SubmitClosedLoopBulkRequest $request
     * @param  null $requestId
     * @param  int $millisecondsToWait
     * @return ErrorResponse|SubmitClosedLoopBulkResponse|static
     * @throws ResponseException
     */
    public function submitClosedLoopBulk(SubmitClosedLoopBulkRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitClosedLoopBulk', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopBulkResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop order anonymously.
     *
     * @param  SubmitClosedLoopAnonymousRequest $request
     * @param  string                           $requestId
     * @param  int                              $millisecondsToWait
     * @return SubmitClosedLoopAnonymousResponse|ErrorResponse
     * @throws ResponseException
     */
    public function submitClosedLoopAnonymous(SubmitClosedLoopAnonymousRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitClosedLoopAnonymous', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopAnonymousResponse::fromHttpResponse($response);
    }

    /**
     * Build a POST request.
     *
     * @param  string  $method
     * @param  Request $request
     * @param  string  $requestId
     * @param  int     $millisecondsToWait
     * @return \Psr\Http\Message\ResponseInterface
     */
    protected function buildPostRequest($method, Request $request, $requestId, $millisecondsToWait)
    {
        $this->incrementAttempts(__FUNCTION__);

        logger('Request data', $request->toArray());

        return $this->request('post', $method, [
            'cert' => $this->ssl->getCertPath(),
            'ssl_key' => $this->ssl->getKeyPath(),
            'json' => $request->toArray(),
            'headers' => $this->getDefaultHeaders([
                'RequestId' => $requestId,
                'millisecondsToWait' => $millisecondsToWait,
                'previousAttempts' => $this->getAttempts(__FUNCTION__),
            ])
        ]);
    }

    /**
     * Make a request to the API.
     *
     * @param  string $method
     * @param  string $uri
     * @param  array  $options
     * @return \Psr\Http\Message\ResponseInterface
     */
    protected function request($method, $uri, array $options)
    {
        return $this->client->request($method, $uri, $options);
    }

    /**
     * The default headers and possibly merge in additional headers.
     *
     * @param  array $additional
     * @return array
     */
    protected function getDefaultHeaders($additional = [])
    {
        return array_merge([
            'MerchantId' => $this->merchantId, // Identifier issued by BHN [string] {required}
        ], $additional);
    }

    /**
     * The default configured HTTP client.
     *
     * @return Client
     */
    protected function getDefaultClient()
    {
        return new Client([
            'base_uri' => config('app.env') === 'production' ? $this->liveEndpoint : $this->testEndpoint,
            'debug' => config('catalog.vendors.hawk.debug_request', false),
            'verify' => false,
            'exceptions' => false,
            'curl' => [
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSLVERSION => CURL_SSLVERSION_TLSv1_2,
                CURLOPT_SSLCERT => config('catalog.vendors.hawk.cert'),
                CURLOPT_SSLKEY => config('catalog.vendors.hawk.key'),
            ]
        ]);
    }
}
