<?php

namespace Ignite\Catalog\Vendors;

use Illuminate\Support\Collection;
use Ignite\Catalog\Contracts\VendorProcessor;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\Vendor;

abstract class Processor implements VendorProcessor
{
    /**
     * @var Vendor
     */
    protected $vendor;

    /**
     * @var array
     */
    protected $actions = [];

    /**
     * Factory constructor.
     *
     * @param Vendor $vendor
     */
    public function __construct(Vendor $vendor)
    {
        $this->vendor = $vendor;
    }

    /**
     * Make a vendor service object.
     *
     * @param  Order $order
     * @param  Collection $items
     * @return void
     */
    public function process(Order $order, Collection $items)
    {
        foreach ($items as $name => $orderItems) {
            if (! $this->hasProcessor($name)) {
                throw new \DomainException(sprintf('%s does not support %s', ucfirst($this->vendor->name), $name));
            }

            $this->resolveProcessor($name)->handle($order, $orderItems);
        }
    }

    /**
     * Retrieve the specific actions for the current processor instance.
     *
     * @return array
     */
    protected function getActions()
    {
        return $this->actions;
    }

    /**
     * Determine if the vendor processor supports the provided action.
     *
     * @param  string $name
     * @return bool
     */
    protected function hasProcessor($name)
    {
        return array_key_exists($name, $this->getActions());
    }

    /**
     * Get the vendor processor fully qualified class name.
     *
     * @param  string $name
     * @return string
     */
    protected function getProcessor($name)
    {
        $actions = $this->getActions();

        return $actions[$name];
    }

    /**
     * Resolve an instance the vendor processor out of the IoC container.
     *
     * @param  string $name
     * @return \Ignite\Catalog\Contracts\VendorHandler
     */
    protected function resolveProcessor($name)
    {
        return app($this->getProcessor($name), ['vendor' => $this->vendor]);
    }
}
