<?php

namespace Ignite\Catalog\Models\Dashboard\Charts;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Ignite\Core\Models\Dashboard\Line;

class OrdersTypeLine extends Line
{
    public function __construct($attributes = [])
    {
        parent::__construct(array_merge([
            'icon' => 'ion ion-bag',
            'label' => 'Orders By Type',
            'element' => 'orders-chart',
            'colors' => ['#00a65a', '#f39c12', '#dd4b39', '#0073b7', '#605ca8', '#cccccc'],
            'labels' => ['Merchandise', 'Gift Card', 'eGift Card', 'Prepaid', 'Reloadable Card'],
            'xkey' => 'date',
            'ykeys' => ['merchandise', 'giftcard', 'egift', 'prepaid', 'reloadable']
        ], $attributes));
    }

    /**
     * The HTML element ID.
     *
     * @return string
     */
    public function getElement()
    {
        return 'order-types-line';
    }

    /**
     * The chart size in the layout.
     *
     * @return string
     */
    public function getSize()
    {
        return $this->get('size', 'col-xs-12 col-sm-12 col-md-8');
    }

    /**
     * Get the correct data value from storage.
     *
     * @return mixed
     */
    public function getData()
    {
        if (! $this->get('data', false)) {
            $this->attributes['data'] = array_values(
                $this->transformData($this->query())
            );
        }

        return $this->get('data');
    }

    /**
     * Transform the data.
     *
     * @param  Collection $results
     * @return array
     */
    private function transformData(Collection $results)
    {
        $data = [];

        foreach ($results as $result) {
            $date = $result->created_at;
            $data[$date]['date'] = $date;
            $data[$date]['merchandise'] = ($result->class === 'merchandise') ? $result->total : $data[$date]['merchandise'] ?? 0;
            $data[$date]['giftcard'] = ($result->class === 'giftcard') ? $result->total : $data[$date]['giftcard'] ?? 0;
            $data[$date]['egift'] = ($result->class === 'egift') ? $result->total : $data[$date]['egift'] ?? 0;
            $data[$date]['prepaid'] = ($result->class === 'prepaid') ? $result->total : $data[$date]['prepaid'] ?? 0;
            $data[$date]['reloadable'] = ($result->class === 'reloadable') ? $result->total : $data[$date]['reloadable'] ?? 0;
        }

        return $data;
    }

    /**
     * Query the database.
     *
     * @return Collection
     */
    private function query()
    {
        /**
         * @var \Carbon\Carbon $start
         * @var \Carbon\Carbon $end
         */
        list($start, $end) = $this->getDateRange();

        DB::select(DB::raw('SET session sql_mode=""'));

        return DB::query()
            ->select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m-%d") AS created_at'),
                'class',
                DB::raw('COUNT(quantity) as total')
            )
            ->from('catalog_order_item')
            ->where(function ($query) {
                $query->where('class', '=', 'merchandise')
                      ->orWhere('class', '=', 'giftcard')
                      ->orWhere('class', '=', 'egift')
                      ->orWhere('class', '=', 'prepaid')
                      ->orWhere('class', '=', 'reloadable');
            })
            ->whereBetween('created_at', [$start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s')])
            ->groupBy(
                'class',
                DB::raw('DATE_FORMAT(created_at, "%Y-%m-%d")')
            )
            ->orderBy('created_at', 'ASC')
            ->get();
    }
}
