<?php

namespace Ignite\Catalog\Emails;

use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\HtmlString;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Core\Entities\User;

class OrderItemFailure extends Mailable
{
    use Queueable, SerializesModels;

    /** @var Order */
    private $order;

    /** @var OrderItem */
    private $orderItem;

    /**
     * Create a new order submitted message instance.
     *
     * @param Order $order
     * @param OrderItem $orderItem
     */
    public function __construct(Order $order, OrderItem $orderItem)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
    }

    /**
     * Build the message.
     *
     * @return $this
     */
    public function build()
    {
        $data = [
            'order' => $this->order,
            'orderItem' => $this->orderItem,
            'meta' => $this->getFormattedMeta(),
        ];

        $this->to(config('mail.from.address'), config('mail.from.name'));
        $this->subject("Order Item Failure: {$this->orderItem->name} on Order #{$this->order->number}");
        $this->from(config('mail.from.address'), config('mail.from.name'));

        return $this->markdown('emails.orders.items.failure', $data);
    }

    /**
     * The formatted vendor meta data.
     *
     * @return string
     */
    private function getFormattedMeta()
    {
        $meta = $this->orderItem->vendor_meta ?? [];
        $temp = "";

        if (is_array($meta) && ! empty($meta)) {
            foreach ($meta['attempts'] as $requestId => $attempt) {
                $temp .= $this->buildAttemptString($attempt);
            }
        }

        return $temp;
    }

    /**
     * Build the markdown string for the attempt.
     *
     * @param  array $attempt
     * @return HtmlString
     */
    private function buildAttemptString($attempt)
    {
        $temp = "";

        if (isset($attempt['item'])) {
            $temp .= "## Item\r\n\r\n";
            foreach ($attempt['item'] as $key => $value) {
                if (is_array($value)) {
                    if (empty($value)) {
                        continue;
                    }
                    $temp .= $this->markdownList($value);
                } else {
                    $temp .= $this->markdownItem($key, $value);
                }
            }
            $temp .= "\r\n\r\n";
        }

        if (isset($attempt['response'])) {
            $temp .= "## Response\r\n\r\n";
            foreach ($attempt['response'] as $key => $value) {
                if (is_array($value)) {
                    if (empty($value)) {
                        continue;
                    }
                    $heading = $this->titleize($key);
                    $temp .= "### $heading\r\n\r\n";
                    $temp .= $this->markdownList($value);
                } else {
                    $temp .= $this->markdownItem($key, $value);
                }
            }
            $temp .= "\r\n\r\n";
        }

        return new HtmlString($temp);
    }

    /**
     * Create a list of items in markdown.
     *
     * @param  array $array
     * @param  int $level
     * @return string
     */
    private function markdownList(array $array, &$level = 0)
    {
        $list = "";
        foreach ($array as $key => $value) {
            if (is_array($value)) {
                ++$level;
                $list .= $this->markdownList($value, $level) . "\r\n";
            } else {
                $list .= $this->markdownItem($key, $value, $level);
            }
        }
        $list .= "\r\n";

        return $list;
    }

    /**
     * Create a list item in markdown.
     *
     * @param  string $key
     * @param  string $value
     * @param  int    $level
     * @return string
     */
    private function markdownItem($key, $value, $level = 0)
    {
        $key = $this->titleize($key);

        return str_repeat("  ", $level) . "- {$key}: {$value}\r\n";
    }

    /**
     * Create a human readable title from a codified string.
     *
     * @param  string $value
     * @return string
     */
    protected function titleize($value)
    {
        return title_case(str_replace(['-', '_', '.'], ' ', snake_case($value)));
    }
}
