<?php

namespace Ignite\Catalog\Http\Controllers;

use Exception;
use Ignite\Flash\Facades\Flash;
use Illuminate\Http\RedirectResponse;
use Ignite\Catalog\Contracts\FavoriteRepository;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Http\Controllers\Controller;
use Illuminate\View\View;

class FavoritesController extends Controller
{
    /**
     * @var FavoriteRepository
     */
    private $favoriteRepository;

    /**
     * FavoritesController constructor.
     *
     * @param FavoriteRepository $favoriteRepository
     */
    public function __construct(FavoriteRepository $favoriteRepository)
    {
        $this->favoriteRepository = $favoriteRepository;
    }

    /**
     * List all the users' favorite items.
     *
     * @return View
     */
    public function index()
    {
        $items = $this->favoriteRepository->paginate(24);

        return view('catalog.favorites.index', compact('items'));
    }

    /**
     * Add the relationship for the user to favorite the item.
     *
     * @param  Item $item
     * @return RedirectResponse
     */
    public function store(Item $item)
    {
        try {
            $this->favoriteRepository->favorite($item);
            $message = 'Item successfully added to favorites.';
            $status = 'success';
        } catch (Exception $e) {
            $message = 'Unable to favorite the item. Error: ' . $e->getMessage();
            $status = 'error';
        }

        Flash::add($message, $status);

        return redirect()->back();
    }

    /**
     * Destroy the relationship for the user to favorite the item.
     *
     * @param  Item $item
     * @return RedirectResponse
     */
    public function destroy(Item $item)
    {
        try {
            $this->favoriteRepository->unfavorite($item);
            $message = 'Item successfully remove from favorites.';
            $status = 'success';
        } catch (Exception $e) {
            $message = 'Unable remove the item from favorites. Error: ' . $e->getMessage();
            $status = 'error';
        }

        Flash::add($message, $status);

        return redirect()->back();
    }
}
