<?php

namespace Ignite\Catalog\Models\Grid;

use Illuminate\Support\HtmlString;
use Ignite\Core\Models\Grid\Formatter;

class ItemFormatter extends Formatter
{
    /**
     * Format a checkbox for selecting entries in the grid.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function checkbox($model)
    {
        return $this->formatCheckbox($model->getKey());
    }

    /**
     * Custom column for actions.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return HtmlString
     */
    public function actions($model)
    {
        $actions = [
            [
                'icon' => 'eye-open',
                'label' => trans('Core::core.buttons.view'),
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.catalog.items.show', [
                    'number' => $model->getKey()
                ])
            ], [
                'icon' => 'pencil',
                'label' => trans('Core::core.buttons.edit'),
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.catalog.items.edit', [
                    'item' => $model->getKey()
                ])
            ], [
                'icon' => 'circle' . ($model->isActive() ? ' text-red' : ' text-green'),
                'label' => ($model->isActive() ? 'Deactivate' : 'Activate'),
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.catalog.items.toggle', [
                    'item' => $model->getKey(),
                    'field' => 'active',
                    'value' => (int) ! $model->isActive()
                ])
            ], [
                'icon' => 'eye' . ($model->isVisible() ? '-slash' : ''),
                'label' => ($model->isVisible() ? 'Hide' : 'Show') . ' in Catalog',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.catalog.items.toggle', [
                    'item' => $model->getKey(),
                    'field' => 'visibility',
                    'value' => (int) ! $model->isVisible()
                ])
            ], [
                'icon' => 'cloud-' . ($model->isVendorActive() ? 'download' : 'upload'),
                'label' => ($model->isVendorActive() ? 'Disable' : 'Enable') . ' Vendor',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.catalog.items.toggle', [
                    'item' => $model->getKey(),
                    'field' => 'vendor_active',
                    'value' => (int) ! $model->isVendorActive()
                ])
            ]
        ];

        if ($model->isSimple()) {
            $actions[] = [
                'icon' => 'copy',
                'label' => 'Duplicate',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.catalog.items.duplicate', [
                    'item' => $model->getKey()
                ])
            ];
        }

        return $this->buttons($actions);
    }

    /**
     * Format an item image.
     *
     * @param \Ignite\Catalog\Entities\Item $model
     * @return HtmlString
     */
    public function image($model)
    {
        return $this->html(sprintf(
            '<img src="%s" class="img-responsive" />',
            $model->getImageUrl()
        ));
    }

    /**
     * Format the visibility.
     *
     * @param \Ignite\Catalog\Entities\Item $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function visibility($model)
    {
        return $this->html($model->present()->getVisibility());
    }

    /**
     * Format the active column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function active($model)
    {
        return $this->html($model->present()->getActiveStatus());
    }

    /**
     * Format a vendor name with link.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function vendor($model)
    {
        return $this->html(sprintf(
            '<a href="%s">%s</a>',
            route('admin.catalog.vendors.show', $model->vendor->getKey()),
            $model->present()->getVendorName()
        ));
    }

    /**
     * Format the vendor active column.
     *
     * @param \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function isVendorEnabled($model)
    {
        return $this->html($model->present()->getVendorStatus());
    }

    /**
     * Format the msrp column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function msrp($model)
    {
        return $model->msrp;
    }
    /**
     * Format the cost column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function cost($model)
    {
        return $model->cost;
    }

    /**
     * Format the price column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function price($model)
    {
        return $model->price;
    }

    /**
     * Format the price_margin column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function margin($model)
    {
        return $model->present()->getPriceMargin();
    }

    /**
     * Format the price_markup column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function markup($model)
    {
        return $model->present()->getPriceMarkup();
    }

    /**
     * Format the points column.
     *
     * @param  \Ignite\Catalog\Entities\Item $model
     * @return string
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function points($model)
    {
        return $model->present()->getPoints(false);
    }
}
