<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Catalog\Contracts\AttributeRepository;
use Ignite\Catalog\Contracts\CatalogRepository;
use Ignite\Catalog\Entities\Attribute;
use Ignite\Catalog\Http\Forms\AttributeForm;
use Ignite\Catalog\Models\Grid\AttributesTable;
use Ignite\Core\Http\Controllers\Controller;
use Kris\LaravelFormBuilder\FormBuilder;

class AttributesController extends Controller
{
    /**
     * Display all attributes.
     *
     * @param AttributesTable $table
     * @param Request      $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(AttributesTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.attributes.index', compact('status'));
    }

    /**
     * Show an attribute.
     *
     * @param  Attribute $attribute
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function show(Attribute $attribute)
    {
        return view('Catalog::admin.attributes.show', compact('attribute'));
    }

    /**
     * Show the create form for a new attribute.
     *
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function create(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(AttributeForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.attributes.create')
        ]);

        return view('Catalog::admin.attributes.create', compact('form'));
    }

    /**
     * Store a new attribute.
     *
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function store(FormBuilder $formBuilder, AttributeRepository $repository)
    {
        $form = $formBuilder->create(AttributeForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $attribute = $repository->create(collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::attribute.store.success', ['label' => $attribute->label]);
            return redirect()->route('admin.catalog.attributes.edit', $attribute);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashSuccess('Catalog::attribute.store.failure');
            return redirect()->route('admin.catalog.attributes.index');
        }
    }

    /**
     * Show the edit form for an existing attribute.
     *
     * @param  Attribute   $attribute
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function edit(Attribute $attribute, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(AttributeForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.attributes.update', $attribute),
            'model' => $attribute
        ]);

        return view('Catalog::admin.attributes.edit', compact('form', 'attribute'));
    }

    /**
     * Update an existing attribute.
     *
     * @param  int $attribute
     * @param  FormBuilder $formBuilder
     * @param  CatalogRepository $repository
     * @return \Illuminate\Http\Response
     */
    public function update($attribute, FormBuilder $formBuilder, AttributeRepository $repository)
    {
        $form = $formBuilder->create(AttributeForm::class);

        if (! $form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $attribute = $repository->update($attribute, collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::attribute.update.success', ['name' => $attribute->label]);
            return redirect()->route('admin.catalog.attributes.edit', $attribute);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::attribute.update.failure');
            return redirect()->route('admin.catalog.attributes.index');
        }
    }

    /**
     * Delete one record.
     *
     * @param  int $attribute
     * @param  AttributeRepository $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($attribute, AttributeRepository $repository)
    {
        try {
            $attribute = $repository->delete($attribute);
            $status = 'success';
            $message = trans('Catalog::attribute.delete.success', ['label' => $attribute->label]);
        } catch (\Exception $e) {
            $this->logException($e);
            $status = 'error';
            $message = trans('Catalog::attribute.delete.failure');
        }

        if (request()->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        $this->flash($status, $message);

        return redirect()->route('admin.catalog.attributes.index');
    }

    /**
     * Delete multiple records at once.
     *
     * @param  Request $request
     * @param  AttributeRepository $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(Request $request, AttributeRepository $repository)
    {
        $ids = $request->ids;
        $values = explode(',', $ids);

        try {
            $repository->deleteMany($values);
            $this->flashSuccess('Catalog::attribute.delete_many.success', ['ids' => $ids]);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::attribute.delete_many.failure', ['ids' => $ids]);
        }

        return redirect()->back();
    }
}
