<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Exception;
use Ignite\Vendor\Hawk\FundingOrderLine;
use Ignite\Vendor\Hawk\PersonalizedOrderLine;
use Ignite\Vendor\Hawk\PersonalizedRecipient;
use Ignite\Vendor\Hawk\Recipient;
use Ignite\Vendor\Hawk\RecipientAddress;
use Ignite\Vendor\Hawk\Request\SubmitFundingRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\ResponseException;
use Ignite\Vendor\Hawk\ShippingMethod;
use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Ignite\Catalog\Jobs\Hawk\Base;
use Ignite\Vendor\Hawk\PaymentType;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Client as HawkApi;
use Illuminate\Support\Str;

class Reloadable extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Execute the job.
     *
     * Call submitOpenLoopPersonalizedIndividual when creating a new PID
     * Call submitFunding on subsequent reloads to an existing PID
     *  orderDetails => FundingOrderLine
     *
     * @param HawkApi $hawkApi
     *
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle(HawkApi $hawkApi)
    {
        $requestId = $this->buildRequestId();

        try {
            $response = $this->createCardOrReloadCard($hawkApi);

            if (
                $response instanceof SubmitOpenLoopPersonalizedIndividualResponse ||
                $response instanceof SubmitFundingResponse
            ) {
                $this->order->participant->update([
                    'reloadable_created_at' => now()->format('Y-m-d H:i:s'),
                    $this->getIdentityColumn() => $this->order->participant->getAttribute($this->getIdentityColumn())
                ]);
                return $this->complete($response, $requestId);
            }

            if ($response instanceof ErrorResponse) {
                return $this->error($response, $requestId);
            }
        } catch (Exception $e) {
            $response = new ErrorResponse(['exception' => [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode(),
                'trace' => $e->getTraceAsString()
            ]]);

            return $this->error($response, $requestId);
        }
    }

    /**
     * Build the request for the eGift Individual.
     *
     * @return SubmitOpenLoopPersonalizedIndividualRequest
     */
    protected function buildOpenLoopPersonalizedIndividual()
    {
        return SubmitOpenLoopPersonalizedIndividualRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $this->order->number,
            'orderDetails' => $this->buildOrderLines(),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return PersonalizedOrderLine
     */
    protected function buildOrderLine()
    {
        return PersonalizedOrderLine::fromAssociativeArray([
            'amount' => (int) $this->orderItem->points,
            'description' => config('catalog.vendors.hawk.reloadable.description', ''),
            'fourthLineEmbossText' => config('catalog.vendors.hawk.reloadable.fourth_line_emboss', 'THANK YOU'),
            'cardCarrierMessage' => config('catalog.vendors.hawk.reloadable.card_carrier_message', ''),
            'shippingMethod' => config('catalog.vendors.hawk.reloadable.shipping_method', ShippingMethod::USPS_STANDARD_MAIL),
            'recipient' => $this->getRecipientData(),
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return PersonalizedRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return PersonalizedRecipient::fromAssociativeArray([
            'id' => $this->order->participant->getAttribute($this->getIdentityColumn()),
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'phoneNumber' => $this->order->ship_phone,
            'address' => RecipientAddress::fromAssociativeArray([
                'line1' => $this->order->ship_address_1,
                'line2' => $this->order->ship_address_2,
                'line3' => $this->order->ship_address_3,
                'city' => $this->order->ship_city,
                'region' => $this->order->ship_state,
                'postalCode' => $this->order->ship_postal,
                'country' => $this->getCountry(),
            ])
        ]);
    }

    /**
     * @param HawkApi $hawkApi
     *
     * @return ErrorResponse|SubmitFundingResponse|SubmitOpenLoopPersonalizedIndividualResponse
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws ResponseException
     * @throws Exception
     */
    protected function createCardOrReloadCard(HawkApi $hawkApi)
    {
        if ($this->hasCard()) {
            return $hawkApi->submitFunding(
                $this->buildSubmitFunding(),
                $this->buildRequestId()
            );
        } else {
            return $hawkApi->submitOpenLoopPersonalizedIndividual(
                $this->buildOpenLoopPersonalizedIndividual(),
                $this->buildRequestId()
            );
        }
    }

    /**
     * Determine whether the participant has a proxy number and PID.
     *
     * @return bool
     * @throws Exception
     */
    protected function hasCard()
    {
        if (! $this->order->user_id) {
            throw new Exception('No user associated with the order.');
        }

        if (! $this->order->participant) {
            $this->order->load('participant');
        }

        if (! $this->order->participant) {
            throw new \Exception('Unable to load the participant associated with the order.');
        }

        return (bool) $this->order->participant->getAttribute($this->getReloadablePidColumn());
    }

    /**
     * Build the submit funding request.
     *
     * @return SubmitFundingRequest
     */
    protected function buildSubmitFunding()
    {
        return SubmitFundingRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $this->order->number,
            'orderDetails' => [$this->buildFundingLines()],
        ]);
    }

    /**
     * Build the FundingOrderLine.
     *
     * @return FundingOrderLine
     */
    protected function buildFundingLines()
    {
        return FundingOrderLine::fromAssociativeArray([
            'amount' => (int) $this->orderItem->points,
            'recipient' => Recipient::fromAssociativeArray([
                'id' => $this->order->participant->getAttribute($this->getReloadablePidColumn()),
            ])
        ]);
    }

    /**
     * Get the product ID for the current environment.
     *
     * @return string
     */
    protected function getProductId()
    {
        if ('production' !== config('app.env')) {
            return config('catalog.vendors.hawk.test_reloadable_id', '43531408');
        } else {
            return $this->orderItem->item->vendor_meta['product_id'];
        }
    }

    /**
     * Format the country.
     *
     * @return string|null
     */
    protected function getCountry()
    {
        if (Str::upper($this->order->ship_country) === 'US') {
            return 'USA';
        }

        if (Str::upper($this->order->ship_country) === 'CA') {
            return 'CAN';
        }

        throw new \DomainException("Unsupported country: {$this->order->ship_country}");
    }

    /**
     * The column on the participant table to identify the user.
     *
     * @return string|int
     */
    protected function getIdentityColumn()
    {
        return 'user_id';
    }

    /**
     * The column on the participant table to identify the reloadable pid.
     *
     * @return string|int
     */
    protected function getReloadablePidColumn()
    {
        return 'reloadable_pid';
    }
}
