<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Http\Forms\CategoryForm;
use Ignite\Catalog\Models\Grid\CategoriesTable;
use Ignite\Catalog\Contracts\CategoryRepository;
use Ignite\Core\Http\Controllers\Controller;
use Kris\LaravelFormBuilder\FormBuilder;

class CategoriesController extends Controller
{
    /**
     * Display all categories.
     *
     * @param CategoriesTable $table
     * @param Request      $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(CategoriesTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.categories.index', compact('status'));
    }

    /**
     * Show a category.
     *
     * @param  Category $category
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function show(Category $category)
    {
        return view('Catalog::admin.categories.show', compact('category'));
    }

    /**
     * Show the create form for a new attribute.
     *
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function create(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(CategoryForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.categories.create')
        ]);

        return view('Catalog::admin.categories.create', compact('form'));
    }

    /**
     * Store a new attribute.
     *
     * @param  FormBuilder $formBuilder
     * @param  CategoryRepository  $repository
     * @return \Illuminate\Http\Response
     */
    public function store(FormBuilder $formBuilder, CategoryRepository $repository)
    {
        $form = $formBuilder->create(CategoryForm::class);

        if (! $form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $category = $repository->create(collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::category.store.success', ['name' => $category->name]);
            return redirect()->route('admin.catalog.categories.edit', $category);
        } catch (\Exception $e) {
            $this->logException($e);
            flash()->error(trans('Catalog::category.store.failure'));
            return redirect()->route('admin.catalog.categories.index');
        }
    }

    /**
     * Show the edit form for an existing attribute.
     *
     * @param  Category    $category
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function edit(Category $category, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(CategoryForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.categories.update', $category),
            'model' => $category
        ]);

        return view('Catalog::admin.categories.edit', compact('form', 'category'));
    }

    /**
     * Update an existing attribute.
     *
     * @param  int $category
     * @param  FormBuilder $formBuilder
     * @param  CategoryRepository  $repository
     * @return \Illuminate\Http\Response
     */
    public function update($category, FormBuilder $formBuilder, CategoryRepository $repository)
    {
        $form = $formBuilder->create(CategoryForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $category = $repository->update($category, collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::category.update.success', ['name' => $category->name]);
            return redirect()->route('admin.catalog.categories.edit', $category);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::category.update.failure');
            return redirect()->route('admin.catalog.categories.index');
        }
    }

    /**
     * Delete one record.
     *
     * @param  int  $category
     * @param  CategoryRepository  $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($category, CategoryRepository $repository)
    {
        try {
            $category = $repository->delete($category);
            $status = 'success';
            $message = trans('Catalog::category.delete.success', ['name' => $category->name]);
        } catch (\Exception $e) {
            $this->logException($e);
            $status = 'error';
            $message = trans('Catalog::category.delete.failure');
        }

        if (request()->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        $this->flash($status, $message);

        return redirect()->route('admin.catalog.categories.index');
    }

    /**
     * Delete multiple records at once.
     *
     * @param  CategoryRepository  $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(CategoryRepository $repository)
    {
        $ids = request()->ids;
        $categories = explode(',', request()->ids);

        try {
            $repository->deleteMany($categories);
            $this->flashSuccess('Catalog::category.delete_many.success', compact('ids'));
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::category.delete_many.failure');
        }

        return redirect()->back();
    }
}
