<?php

namespace Ignite\Catalog\Http\Controllers;

use Ignite\Core\Contracts\Repositories\ParticipantRepository;
use Illuminate\Support\Facades\Auth;
use Ignite\Catalog\Contracts\OrderRepository;
use Ignite\Catalog\Http\Requests\OrderCreateRequest;
use Ignite\Catalog\Models\Cart;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Entities\Participant;

class CheckoutController extends Controller
{
    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * @var ParticipantRepository
     */
    private $participantRepository;

    /**
     * CheckoutController constructor.
     *
     * @param OrderRepository $orderRepository
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(OrderRepository $orderRepository, ParticipantRepository $participantRepository)
    {
        $this->orderRepository = $orderRepository;
        $this->participantRepository = $participantRepository;
    }

    /**
     * Create a new checkout request.
     *
     * @param  Cart $cart
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function create(Cart $cart)
    {
        if (! $cart->canCheckout() || $cart->empty()) {
            return redirect()->route('catalog.cart.index');
        }

        return view('catalog.checkout.create', compact('cart'));
    }

    /**
     * Store the order and order items and create a transaction.
     *
     * @param  OrderCreateRequest $request
     * @param  Cart $cart
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Exception
     */
    public function store(OrderCreateRequest $request, Cart $cart)
    {
        if (! $cart->canCheckout() || $cart->empty()) {
            return redirect()->route('catalog.cart.index');
        }

        try {
            $participant = $this->participantRepository->findAuthenticated();
            $order = $this->orderRepository->create([
                'user_id' => auth()->user()->getKey(),
                'quantity' => $cart->quantity(),
                'points' => $cart->total(),
                'ship_name' => implode(' ', $request->only('first', 'last')),
                'ship_email' => $request->get('ship_email', $participant->email),
                'ship_phone' => $request->get('ship_phone', $participant->phone1),
                'ship_address_1' => $request->get('ship_address_1', $participant->address_1),
                'ship_address_2' => $request->get('ship_address_2', $participant->address_2),
                'ship_address_3' => $request->get('ship_address_3'),
                'ship_city' => $request->get('ship_city', $participant->city),
                'ship_state' => $request->get('ship_state', $participant->state),
                'ship_postal' => $request->get('ship_postal', $participant->postal),
                'ship_country' => $request->get('ship_country', $participant->country),
            ], $cart->items(), $cart->total());

            flash()->success('Order placed successfully.')->important();

            return redirect()->route('catalog.checkout.show', ['number' => $order->number]);
        } catch (\Exception $e) {
            logger('Unable to process order: ', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode()
            ]);

            flash()->error('There was an error placing your order.' . (config('app.debug') ? ' Error: ' . $e->getMessage() : ''))->important();

            return redirect()->back()->withInput();
        }
    }

    /**
     * Show the checkout order information.
     *
     * @param  string $number
     * @return \Illuminate\View\View
     */
    public function show($number)
    {
        $order = $this->orderRepository->findByLoggedInUser($number);

        return view('catalog.checkout.show', compact('order'));
    }
}
