<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Http\Forms\OrderForm;
use Ignite\Catalog\Http\Requests\OrderDeleteManyRequest;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Catalog\Models\Grid\OrdersTable;
use Ignite\Catalog\Contracts\OrderRepository;
use Ignite\Catalog\Contracts\CatalogRepository;
use Ignite\Catalog\Models\Grid\OrderItemTable;
use Kris\LaravelFormBuilder\FormBuilder;

class OrdersController extends Controller
{
    /**
     * @var CatalogRepository
     */
    protected $catalogRepository;

    /**
     * @var OrderRepository
     */
    protected $orderRepository;

    /**
     * Create a new controller instance.
     *
     * @param CatalogRepository $catalogRepository
     * @param OrderRepository $orderRepository
     */
    public function __construct(
        CatalogRepository $catalogRepository,
        OrderRepository $orderRepository
    ) {
        $this->catalogRepository = $catalogRepository;
        $this->orderRepository = $orderRepository;
    }

    /**
     * Display all orders.
     *
     * @param  OrdersTable $table
     * @param  Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(OrdersTable $table, Request $request)
    {
        $status = $request->get('status', 'all');
        $table->with('status', $status);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        $tabs = $this->getTabData();

        return $table->render('Catalog::admin.orders.index', compact('status', 'tabs'));
    }

    /**
     * The tab labels and counts.
     *
     * @return array
     */
    protected function getTabData()
    {
        $tabs = [
            'all' => ['label' => 'All', 'count' => Order::query()->count()]
        ];

        foreach (['processed', 'processing', 'cancelled'] as $tab) {
            $tabs[$tab] = [
                'label' => ucwords($tab),
                'count' => Order::query()->$tab()->count()
            ];
        }

        return $tabs;
    }

    /**
     * Display an order and order item details.
     *
     * @param  string                $number
     * @param  OrderItemTable $table
     * @param  Request               $request
     * @return \Illuminate\Http\JsonResponse|Response
     */
    public function show($number, OrderItemTable $table, FormBuilder $formBuilder, Request $request)
    {
        // Get the Order with all its details
        $order = $this->orderRepository->find($number);

        $form = $formBuilder->create(OrderForm::class, [
            'method' => 'PUT',
            'url' => route('admin.catalog.orders.update', compact('number')),
            'model' => $order
        ]);

        if (empty($order)) {
            flash()->error("Order #{$number} could not be found")->important();
            return back();
        }

        $referer = request()->headers->get('referer');

        $table->with('order', $order);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.orders.show', compact('order', 'form', 'referer'));
    }

    /**
     * Update order information.
     *
     * @param  int $order
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update($order, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(OrderForm::class);

        try {
            $this->orderRepository->update($order, $form->getFieldValues());
            $this->flashSuccess('Catalog::order.update.success', ['number' => $order]);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::order.update.failure');
        }

        return redirect()->back();
    }

    /**
     * Cancel the entire order and return the participants points.
     *
     * @param  string $number
     * @return \Illuminate\Http\RedirectResponse
     */
    public function cancel($number)
    {
        try {
            $order = $this->orderRepository->cancel($number);
            flash()->success("Order {$order->number} was cancelled successfully.")->important();
        } catch (\Exception $e) {
            $this->logException($e);
            flash()->error("Order could not be cancelled. Reason: " . $e->getMessage())->important();
        }

        return redirect()->back();
    }

    /**
     * Cancel the given orders and return the participants points.
     *
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function cancelMany(Request $request)
    {
        $ids = explode(',', $request->get('ids', ''));
        $orders = $this->orderRepository->query()->whereIn('id', $ids)->get();

        app('db')->transaction(function () use ($orders) {
            foreach ($orders as $order) {
                $this->orderRepository->cancel($order->number);
            }
        });

        return redirect()->back();
    }

    /**
     * Process the entire order including all order items.
     *
     * @param  string $number
     * @return \Illuminate\Http\RedirectResponse
     */
    public function process($number)
    {
        try {
            $order = $this->orderRepository->process($number);
            flash()->success("Order {$order->number} was processed successfully.")->important();
        } catch (\Exception $e) {
            $this->logException($e);
            flash()->error("Order could not be processed. Reason: " . $e->getMessage())->important();
        }

        return redirect()->back();
    }

    /**
     * Process the all of the given orders.
     *
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function processMany(Request $request)
    {
        $ids = explode(',', $request->get('ids', ''));
        $orders = $this->orderRepository->query()->whereIn('id', $ids)->get();

        app('db')->transaction(function () use ($orders) {
            foreach ($orders as $order) {
                $this->orderRepository->process($order->number);
            }
        });

        return redirect()->back();
    }

    /**
     * Delete one record.
     *
     * @param Order $order
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Order $order)
    {
        try {
            $order->forceDelete();
            $status = 'success';
            $message = "Order {$order->number} was deleted successfully.";
        } catch (\Exception $e) {
            $this->logException($e);
            $status = 'error';
            $message = "Unable to delete order: {$order->number}.";
        }

        flash()->message($message, $status)->important();

        return response()->json([
            'status' => $status,
            'message' => $message
        ]);
    }

    /**
     * Delete multiple records at once.
     *
     * @param  OrderDeleteManyRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(OrderDeleteManyRequest $request)
    {
        $ids = $request->ids;

        try {
            $this->orderRepository->delete($ids);
            $this->flashSuccess('Catalog::order.delete_many.success', compact('ids'));
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashSuccess('Catalog::order.delete_many.failure', compact('ids'));
        }

        return redirect()->back();
    }
}
