<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Illuminate\Support\Facades\Mail;
use Ignite\Catalog\Emails\OrderItemFailure;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Vendor\Hawk\Response\BaseOrderResponse;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\Response;

abstract class Base
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderItem
     */
    protected $orderItem;

    /**
     * Create a new job instance.
     *
     * @param \Ignite\Catalog\Entities\Order     $order
     * @param \Ignite\Catalog\Entities\OrderItem $orderItem
     */
    public function __construct(Order $order, OrderItem $orderItem)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
    }

    /**
     * Build up the request id.
     *
     * @return string
     */
    protected function buildRequestId()
    {
        return sprintf(
            '%s_%s_%s',
            $this->order->number,
            $this->orderItem->id,
            $this->orderItem->created_at->format('Y_m_d_H_i_s')
        );
        //return $this->orderItem->getVendorRequestId()
    }

    /**
     * Get the product ID for the current environment.
     *
     * @return string
     */
    protected function getProductId()
    {
        return $this->orderItem->item->vendor_meta['product_id'];
    }

    /**
     * Complete the order.
     *
     * @param  BaseOrderResponse $response
     * @param  string            $requestId
     * @return bool
     */
    protected function complete(BaseOrderResponse $response, $requestId)
    {
        if ($response->getIsCompleted() && $response->getSuccess()) {
            $this->process($response, $requestId, [
                'processed' => 1,
                'processed_at' => now(),
                'processed_quantity' => $this->orderItem->quantity
            ]);
            return true;
        }

        return false;
    }

    /**
     * Report the error.
     *
     * @param  ErrorResponse $response
     * @param  string $requestId
     * @return bool
     */
    protected function error(ErrorResponse $response, $requestId)
    {
        // We're not cancelling the item yet, instead, we'll send an email with the reason.
        $this->process($response, $requestId);

        Mail::send(new OrderItemFailure($this->order, $this->orderItem));

        return true;
    }

    /**
     * Process the response.
     *
     * @param  Response $response
     * @param  string   $requestId
     * @param  array    $data
     * @param  array    $meta
     * @return \Ignite\Catalog\Entities\OrderItem
     */
    protected function process(Response $response, $requestId, array $data = [], array $meta = [])
    {
        $existing = $this->orderItem->vendor_meta ?? [];

        if (! isset($existing['attempts'])) {
            $existing['attempts'] = [];
        }

        $existing['attempts'][$requestId] = [
            'response' => $response->toArray(),
            'item' => $this->orderItem->item->vendor_meta ?? []
        ];

        if (array_key_exists('attempts', $meta)) {
            unset($meta['attempts']);
        }

        $this->orderItem->update(array_merge([
            'vendor_meta' => array_merge($existing, $meta),
        ], $data));

        return $this->orderItem;
    }
}
