<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Ignite\Vendor\Hawk\EGiftIndividualOrderLine;
use Ignite\Vendor\Hawk\EmailContent;
use Ignite\Vendor\Hawk\PaymentType;
use Ignite\Vendor\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\VirtualRecipient;
use Ignite\Vendor\Hawk\ResponseException;
use Ignite\Vendor\Hawk\Client as HawkApi;

class Egift extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Execute the job.
     *
     * @param  HawkApi $hawkApi
     * @return bool
     */
    public function handle(HawkApi $hawkApi)
    {
        $requestId = $this->buildRequestId();

        try {
            $response = $hawkApi->submitEgiftIndividual(
                $this->buildEgiftIndividualRequest(),
                $this->buildRequestId()
            );

            if ($response instanceof SubmitEGiftIndividualResponse) {
                return $this->complete($response, $requestId);
            }

            if ($response instanceof ErrorResponse) {
                return $this->error($response, $requestId);
            }
        } catch (ResponseException $e) {
            $response = new ErrorResponse(['exception' => [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode(),
                'trace' => $e->getTraceAsString()
            ]]);

            return $this->error($response, $requestId);
        }
    }

    /**
     * Build the request for the eGift Individual.
     *
     * @return SubmitEGiftIndividualRequest
     */
    protected function buildEgiftIndividualRequest()
    {
        return SubmitEGiftIndividualRequest::fromAssociativeArray([
            'paymentType' => PaymentType::DRAW_DOWN,
            'returnCardNumberAndPIN' => true,
            'poNumber' => $this->order->number,
            'clientProgramNumber' => $this->getProductId(),
            'orderDetails' => $this->buildOrderLines(),
            'emailContent' => app(EmailContent::class),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return EgiftIndividualOrderLine
     */
    protected function buildOrderLine()
    {
        return EgiftIndividualOrderLine::fromAssociativeArray([
            'recipient' => $this->getRecipientData(),
            'amount' => (int) $this->orderItem->points,
            'contentProvider' => $this->orderItem->item->vendor_meta['content_provider_code'],
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return VirtualRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return VirtualRecipient::fromAssociativeArray([
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'id' => $this->order->user_id
        ]);
    }
}
