<?php

namespace Ignite\Catalog\Entities;

use Ignite\Core\Facades\Format;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Ignite\Core\Entities\User;

/**
 * Cart Item Entity
 *
 * @property int id
 * @property int user_id
 * @property int catalog_id
 * @property int item_id
 * @property int super_id
 * @property int quantity
 * @property int price
 * @property int points
 * @property Item item
 * @property Item super
 */
class CartItem extends Model
{
    /** @var string */
    protected $primaryKey = 'id';

    /** @var string */
    protected $table = 'catalog_cart_item';

    /** @var array */
    protected $guarded = array();

    /**
     * The relationship to the specific cart for this line item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function cart()
    {
        return $this->belongsTo(\Ignite\Catalog\Entities\Cart::class, 'catalog_cart_id', 'id');
    }

    /**
     * The relationship to the catalog item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function item()
    {
        return $this->hasOne(Item::class, 'id', 'item_id');
    }

    /**
     * The relationship to the user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function user()
    {
        return $this->hasOne(User::class, 'user_id', 'user_id');
    }

    /**
     * The relationship to the catalog.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function catalog()
    {
        return $this->hasOne(\Ignite\Catalog\Entities\Catalog::class, 'id', 'catalog_id');
    }

    /**
     * The relationship to the vendor.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function vendor()
    {
        return $this->hasOne(\Ignite\Catalog\Entities\Vendor::class, 'id', 'catalog_vendor_id');
    }

    /**
     * The relationship to the user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function super()
    {
        return $this->hasOne(Item::class, 'id', 'super_id');
    }

    /**
     * Scope a query by attaching a user.
     *
     * @param  Builder  $query
     * @param  User|int $user
     * @return Builder
     */
    public function scopeByUser(Builder $query, $user)
    {
        $user = ($user instanceof User) ? $user->getKey() : $user;
        $query->where('user_id', $user);

        return $query;
    }

    /**
     * Scope a query by attaching the current user.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeByCurrentUser(Builder $query)
    {
        $query->where('user_id', auth()->user()->getKey());

        return $query;
    }

    /**
     * Scope a query by attaching the catalog item.
     *
     * @param  Builder  $query
     * @param  Item|int $item
     * @return Builder
     */
    public function scopeForItem(Builder $query, $item)
    {
        $item = ($item instanceof Item) ? $item->getKey() : $item;
        $query->where('item_id', $item);

        return $query;
    }

    /**
     * Scope a query by attaching the catalog items.
     *
     * @param  Builder $query
     * @param  array   $items
     * @return Builder
     */
    public function scopeForItems(Builder $query, array $items)
    {
        $query->whereIn('item_id', $items);

        return $query;
    }

    /**
     * Determine if the item in the cart is merchandise.
     *
     * @return bool
     */
    public function isMerchandise()
    {
        return $this->item->isMerchandise();
    }

    /**
     * Determine if the item in the cart is a gift card.
     *
     * @return bool
     */
    public function isCard()
    {
        return $this->item->isCardType();
    }

    /**
     * Determine if the item in the cart is an eGift card.
     *
     * @return bool
     */
    public function isEgiftCard()
    {
        return $this->item->isEgift();
    }

    /**
     * Determine if the item in the cart is a prepaid card.
     *
     * @return bool
     */
    public function isPrepaidCard()
    {
        return $this->item->isPrepaid();
    }

    /**
     * Determine if the item in the cart is a reloadable card.
     *
     * @return bool
     */
    public function isReloadableCard()
    {
        return $this->item->isReloadable();
    }

    /**
     * The item id.
     *
     * @return int
     */
    public function getItemId()
    {
        return $this->super ? $this->super->getKey() : $this->item->getKey();
    }

    /**
     * The url of the item.
     *
     * @return string
     */
    public function getItemUrl()
    {
        return $this->super
            ? $this->super->getUrl(['linked' => $this->item->getKey()])
            : $this->item->getUrl();
    }

    /**
     * The image url for the item.
     *
     * @return string
     */
    public function getItemImageUrl()
    {
        return $this->item->getImageUrl();
    }

    /**
     * The item name.
     *
     * @return string
     */
    public function getItemName()
    {
        return $this->item->name;
    }

    /**
     * The item sku.
     *
     * @return string
     */
    public function getItemSku()
    {
        return $this->item->sku;
    }

    /**
     * The item quantity.
     *
     * @return string
     */
    public function getItemQuantity()
    {
        return number_format($this->quantity);
    }

    /**
     * The item total.
     *
     * @return string
     */
    public function getItemTotal()
    {
        return Format::amount($this->quantity * $this->points);
    }

    /**
     * Get the URL to delete the item.
     *
     * @return string
     */
    public function getRemoveUrl()
    {
        return route('catalog.cart.destroy', ['id' => $this->item_id]);
    }
}
