<?php

namespace Ignite\Catalog\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\HtmlString;
use Ignite\Catalog\Contracts\CartRepository;
use Ignite\Catalog\Http\Requests\CartItemUpdateRequest;
use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Models\Product\BuyRequest;
use Ignite\Core\Http\Controllers\Controller;

class CartController extends Controller
{
    /**
     * @var CartRepository
     */
    private $cartRepository;

    /**
     * CartController constructor.
     *
     * @param CartRepository $cartRepository
     */
    public function __construct(CartRepository $cartRepository)
    {
        $this->cartRepository = $cartRepository;
    }

    /**
     * The index action lists the items in the cart.
     *
     * @param  Cart $cart
     * @return \Illuminate\View\View
     */
    public function index(Cart $cart)
    {
        return view('catalog.cart.index', compact('cart'));
    }

    /**
     * The store action adds an item to the user's cart.
     *
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        try {
            $buyRequest = BuyRequest::fromHttpRequest($request);

            $this->cartRepository->add($buyRequest);

            flash()->success(sprintf(
                '%s was added to your cart. %s',
                $buyRequest->getItem()->name,
                new HtmlString(sprintf('<a href="%s" class="btn btn-sm btn-default btn-alert">Checkout Now</a>', route('catalog.cart.index')))
            ))->important();

            return redirect()->back();//->route('catalog.cart.index');
        } catch (\DomainException $exception) {
            flash()->error($exception->getMessage())->important();

            return redirect()->back();
        }
    }

    /**
     * The update action updates quantities of items in the cart.
     *
     * @param  CartItemUpdateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CartItemUpdateRequest $request)
    {
        $updates = $request->get('items', []);

        try {
            $this->cartRepository->updateQuantities($updates);
            flash()->success('Your cart was updated successfully.')->important();
        } catch (\Exception $e) {
            logger($e->getMessage(), $e);
            flash()->error('Unable to update your cart.')->important();
        }

        return redirect()->route('catalog.cart.index');
    }

    /**
     * The destroy action removes an item from the user's cart.
     *
     * @param  Request $request
     * @param  int|null $id
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function destroy(Request $request, $id = null)
    {
        try {
            $cart = $this->cartRepository->remove($request->get('catalog_item_id', $id));
            $status = 'success';
            $message = sprintf("%s was removed from cart.", $cart->item->name);
        } catch (\Exception $e) {
            $status = 'error';
            $message = sprintf("Error: %s", $e->getMessage());
        }

        if ($request->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        flash()->message($message, $status)->important();

        return redirect()->route('catalog.cart');
    }
}
