<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Catalog\Contracts\CatalogRepository;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Http\Forms\CatalogForm;
use Ignite\Catalog\Http\Requests\CatalogDeleteManyRequest;
use Ignite\Catalog\Models\Grid\CatalogItemTable;
use Ignite\Catalog\Models\Grid\CatalogsTable;
use Ignite\Core\Http\Controllers\Controller;
use Kris\LaravelFormBuilder\FormBuilder;

class CatalogController extends Controller
{
    /**
     * Display all catalogs.
     *
     * @param  CatalogsTable  $table
     * @param  Request  $request
     * @param  string  $status
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(CatalogsTable $table, Request $request, $status = 'active')
    {
        $status = strtolower($status);
        $table->with('status', $status);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.catalog.index', compact('status'));
    }

    /**
     * Show a catalog.
     *
     * @param  Catalog  $catalog
     * @param  CatalogItemTable  $table
     * @param  Request  $request
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function show(Catalog $catalog, CatalogItemTable $table, Request $request)
    {
        $table->with('catalog', $catalog);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.catalog.show', compact('status', 'catalog'));
    }

    /**
     * Show the create form for a new catalog.
     *
     * @param  FormBuilder  $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function create(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(CatalogForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.store')
        ]);

        return view('Catalog::admin.catalog.create', compact('form'));
    }

    /**
     * Store a new catalog.
     *
     * @param  FormBuilder  $formBuilder
     * @param  CatalogRepository  $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(FormBuilder $formBuilder, CatalogRepository $repository)
    {
        $form = $formBuilder->create(CatalogForm::class);

        if (! $form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $catalog = $repository->create(collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::catalog.store.success', ['name' => $catalog->name]);
            return redirect()->route('admin.catalog.edit', $catalog->code);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::catalog.store.failure');
            return redirect()->route('admin.catalog.index');
        }
    }

    /**
     * Show the edit form for an existing catalog.
     *
     * @param  Catalog     $catalog
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\RedirectResponse
     */
    public function edit(Catalog $catalog, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(CatalogForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.update', $catalog),
            'model' => $catalog
        ]);

        return view('Catalog::admin.catalog.edit', compact('catalog', 'form'));
    }

    /**
     * Update an existing catalog.
     *
     * @param  int  $catalog
     * @param  FormBuilder  $formBuilder
     * @param  CatalogRepository  $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update($catalog, FormBuilder $formBuilder, CatalogRepository $repository)
    {
        $form = $formBuilder->create(CatalogForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $catalog = $repository->update($catalog, collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::catalog.update.success', ['name' => $catalog->name]);
            return redirect()->route('admin.catalog.edit', $catalog->code);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashSuccess('Catalog::catalog.update.failure');
            return redirect()->route('admin.catalog.index');
        }
    }

    /**
     * Delete one record.
     *
     * @param  int $catalog
     * @param  CatalogRepository $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($catalog, CatalogRepository $repository)
    {
        try {
            $catalog = $repository->delete($catalog);
            $status = 'success';
            $message = trans('Catalog::catalog.delete.success', ['name' => $catalog->name]);
        } catch (\Exception $e) {
            $this->logException($e);
            $status = 'error';
            $message = trans('Catalog::catalog.delete.failure');
        }

        if (request()->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        $this->flash($status, $message);

        return redirect()->route('admin.catalog.index');
    }

    /**
     * Delete multiple records at once.
     *
     * @param  CatalogDeleteManyRequest $request
     * @param  CatalogRepository $catalogRepository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(CatalogDeleteManyRequest $request, CatalogRepository $catalogRepository)
    {
        $ids = $request->ids;
        $values = explode(',', $ids);

        try {
            $catalogRepository->deleteMany($values);
            $this->flashSuccess('Catalog::catalog.delete_many.success', ['ids' => $ids]);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::catalog.delete_many.failure', ['ids' => $ids]);
        }

        return redirect()->route('admin.catalog.index');
    }
}
