<?php

namespace Ignite\Catalog\Console;

use Ignite\Core\Facades\Format;
use Illuminate\Console\Command;
use Ignite\Catalog\Entities\Attribute;
use Ignite\Catalog\Entities\AttributeItem;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Menu;
use Ignite\Catalog\Entities\MenuItem;
use Ignite\Catalog\Entities\Option;
use Ignite\Catalog\Entities\Vendor;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

class Brightspot extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:catalog:brightspot';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import Brightspot (demo) catalog data into Ignite';

    /**
     * Flag to determine if we are in dry run mode.
     *
     * @var bool
     */
    protected $isDryRun;

    /**
     * Positive values for the dry run option.
     *
     * @var array
     */
    protected $dryRunPositiveValues = [
        true, 'true', 1, '1', 'yes', 'y', null
    ];

    /**
     * The collection of categories.
     *
     * @var \Illuminate\Support\Collection
     */
    protected $categories;

    /**
     * The catalog to attach the items.
     *
     * @var Catalog
     */
    protected $catalog;

    /**
     * The catalog vendor to attach the items.
     *
     * @var Vendor
     */
    protected $vendor;

    /**
     * The catalog menu to attach the items.
     *
     * @var Menu
     */
    protected $menu;

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $filename = $this->option('file');

        $data = $this->getJsonData($filename);

        if (empty($data)) {
            throw new \DomainException("No data available in $filename.");
        }

        return $this->process($data);
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        if (is_null($this->isDryRun)) {
            $this->isDryRun = in_array($this->option('dry-run'), $this->dryRunPositiveValues, true);
        }
        return $this->isDryRun;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['point', null, InputOption::VALUE_OPTIONAL, 'The value of a single point.', null],
            ['markup', null, InputOption::VALUE_OPTIONAL, 'The value of a single point.', null],
            ['file', null, InputOption::VALUE_OPTIONAL, 'The file from which to import card data.', realpath(__DIR__ . "/data/brightspot/merchandise.json")],
            ['dry-run', null, InputOption::VALUE_OPTIONAL, 'Should we run the import or dry-run it?', false]
        ];
    }

    /**
     * Handle an exception depending on the command verbosity.
     *
     * @param $e
     */
    protected function handleException($e)
    {
        if ($this->output->isVerbose() || $this->output->isVeryVerbose() || $this->output->isDebug()) {
            throw $e;
        }

        $this->error(
            sprintf('An error occurred: %s in %s on line %s', $e->getMessage(), $e->getFile(), $e->getLine())
        );
    }

    /**
     * The data from the JSON file.
     *
     * @param  string $filename
     * @return array
     */
    private function getJsonData($filename)
    {
        if (! starts_with($filename, '/')) {
            $filename = realpath(__DIR__ . "/data/brightspot/$filename");
        }

        $json = file_get_contents($filename);

        $data = json_decode($json, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException("Unable to parse json in $filename. Error: " . json_last_error_msg());
        }

        if (! isset($data['data']) || ! is_array($data['data'])) {
            throw new \DomainException("Unable to find expected top-level key named 'data'.");
        }

        return $data['data'];
    }

    /**
     * Process the items.
     *
     * @param  array $data
     * @return bool
     */
    private function process($data)
    {
        // Wrap the array data in a collection instance.
        $rows = collect($data);
        $this->info(sprintf('Importing %s %s.', $count = $rows->count(), str_plural('item', $count)));

        // Set up an empty categories collection.
        $categories = collect();

        // Set up an empty confirgurable product collection
        $configurables = collect();

        // Create the vendor
        $vendor = $this->getOrCreateVendor();

        // Create the catalog
        $catalog = $this->getOrCreateCatalog($vendor);

        // Create the menu
        $menu = $this->getOrCreateMenu($catalog);

        // Attach the product to the vendor, catalog, category and menu
        foreach ($data as $number => $row) {
            // Extract the categories
            if (isset($row['categories']) && ! empty($row['categories'])) {
                $this->categories(
                    $categories = $this->updateOrCreateCategories($row['categories'])->keyBy('code')
                );
                unset($row['categories']);
            }

            // Extract the configurables
            if (isset($row['type']) && $row['type'] == 'configurable') {
                $configurables->push($row);
                unset($row['attribute']);
            }

            // Create the item
            $item = $this->createItem($row, $vendor, $catalog);

            $item->categories()->detach($categories);
            foreach ($categories as $category) {
                $item->categories()->attach($category->getKey(), [
                    'position' => $number + 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            if (! $this->isDryRun()) {
                $item->save();
            }
        }

        foreach ($configurables as $configurable) {
            // Look up items by sku for a match
            $results = Item::query()->where('sku', 'like', "{$configurable['sku']}_%")->get();
            $options = [];

            foreach ($results as $index => $result) {
                // Attach Options
                $label = ucwords(str_replace('_', ' ', substr($result->sku, strlen($configurable['sku']) + 1)));
                $options[] = [
                    [
                        'position' => $index + 1,
                        'value' => $result->points,
                        'label' => ends_with($label, 'gb') ? strtoupper($label) : $label,
                        'item_id' => $result->getKey(),
                        'attribute_id' => $this->getOrCreateAttribute($configurable['attribute'])->getKey()
                    ]
                ];
            }

            unset($configurable['attribute']);

            /** @var Item $configurable */
            $configurable = Item::firstOrCreate(['sku' => $configurable['sku']], $configurable);
            $configurable->attachOptions($options);
        }

        $position = 0;
        foreach ($categories->values() as $category) {
            MenuItem::updateOrCreate([
                'catalog_category_id' => $category->getKey(),
                'catalog_menu_id' => $menu->getKey(),
            ], [
                'catalog_category_id' => $category->getKey(),
                'catalog_menu_id' => $menu->getKey(),
                'active' => 1,
                'parent_id' => 0,
                'position' => $position + 1
            ]);
        }
    }

    /**
     * Update or create categories.
     *
     * @param  string $categories
     * @return \Illuminate\Support\Collection
     */
    private function updateOrCreateCategories($categories)
    {
        return collect(explode('|', $categories))->map(function ($category) {
            $code = $this->getCategoryCode($category);
            return Category::updateOrCreate(['code' => $code], [
                'code' => $code,
                'name' => $category,
                'active' => 1
            ]);
        });
    }

    /**
     * Get the category code from the category name.
     *
     * @param  string $category
     * @return string
     */
    private function getCategoryCode($category)
    {
        return str_slug(str_replace(['&', '+'], 'and', $category));
    }

    /**
     * Get or merge the categories.
     *
     * @param  null|\Illuminate\Support\Collection $items
     * @return static
     */
    private function categories($items = null)
    {
        if (is_null($this->categories)) {
            $this->categories = collect();
        }

        if (! is_null($items)) {
            foreach ($items as $item) {
                $this->categories->push($item);
            }
        }

        return $this->categories->keyBy('code');
    }

    /**
     * Create an item.
     *
     * @param  array   $row
     * @param  Vendor  $vendor
     * @param  Catalog $catalog
     * @return Item
     */
    private function createItem($row, $vendor, $catalog)
    {
        if (! isset($row['catalog_vendor_id'])) {
            $row['catalog_vendor_id'] = $vendor->getKey();
        }

        if (! isset($row['catalog_id'])) {
            $row['catalog_id'] = $catalog->getKey();
        }

        if (! isset($row['msrp'])) {
            $row['msrp'] = 0.0000;
        }

        if (! isset($row['cost'])) {
            $row['cost'] = 0.0000;
        }

        if (! isset($row['price'])) {
            $row['price'] = 0.0000;
        }

        if (! isset($row['points'])) {
            $row['points'] = 0.0000;
        }

        if (! isset($row['vendor_meta'])) {
            $row['vendor_meta'] = [];
        }

        if ((float) $row['cost'] > 0) {
            $row['points'] = $this->calculatePoints(
                $row['cost'],
                $row['price_markup'],
                $row['point_value']
            );
        }

        return Item::updateOrCreate(['sku' => $row['sku']], $row);
    }

    /**
     * Calculate the points from the price, markup and point value.
     *
     * @param  float $cost
     * @param  float $markup
     * @param  float $point
     * @return float
     */
    private function calculatePoints($cost, $markup, $point)
    {
        /**
         * "price": 1148.65,
         * "price_markup": 0.15,
         * "price_margin": 0.1304,
         * "point_value": 0.05,
         * "points": 22977.00,
         */
        return ($cost * (1 + $markup)) / $point;
    }

    /**
     * Create the option.
     *
     * @param  Item      $item
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     *
     * @return Option
     */
    private function createOption($item, $simple, $attribute, $index)
    {
        $value = Format::amount($simple->price);

        $where = [
            'super_id' => $item->getKey(),
            'item_id' => $simple->getKey(),
            'attribute_id' => $attribute->getKey(),
            'value' => $value,
        ];

        $data = array_merge($where, [
            'label' => '$' . $value,
            'position' => $index + 1
        ]);

        return Option::updateOrCreate($where, $data);
    }

    /**
     * Create the association to the attribute.
     *
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     * @return AttributeItem
     */
    private function createAttributeItem($simple, $attribute, $index)
    {
        $value = Format::amount($simple->price);

        $where = [
            'attribute_id' => $attribute->getKey(),
            'item_id' => $simple->getKey(),
            'value' => $value
        ];

        $data = [
            'position' => $index + 1,
            'active' => 1
        ];

        return AttributeItem::updateOrCreate($where, $data);
    }

    /**
     * The vendor to attach items.
     *
     * @return Vendor
     */
    private function getOrCreateVendor()
    {
        if (is_null($this->vendor)) {
            $this->vendor = Vendor::firstOrCreate(
                ['name' => config('catalog.vendors.brightspot.name')],
                config('catalog.vendors.brightspot')
            );
        }

        return $this->vendor;
    }

    /**
     * The catalog to attach items.
     *
     * @param  Vendor $vendor
     * @return Catalog
     */
    private function getOrCreateCatalog(Vendor $vendor)
    {
        if (is_null($this->catalog)) {
            $this->catalog = Catalog::firstOrCreate(
                ['code' => config('catalog.brightspot.code')],
                array_merge(config('catalog.brightspot'), [
                    "catalog_vendor_id" => $vendor->getKey(),
                ])
            );
        }

        return $this->catalog;
    }

    /**
     * The menu to attach categories.
     *
     * @param  Catalog $catalog
     * @return Menu
     */
    private function getOrCreateMenu(Catalog $catalog)
    {
        if (is_null($this->menu)) {
            $name = "Brightspot Merchandise Menu";
            $this->menu = Menu::firstOrCreate([
                'catalog_id' => $catalog->getKey(),
                "name" => $name
            ], [
                "catalog_id" => $catalog->getKey(),
                "name" => $name,
                "active" => 1
            ]);
        }

        return $this->menu;
    }

    /**
     * Ensure we have a denomination attribute.
     *
     * @param  string $attribute
     * @return Attribute
     */
    private function getOrCreateAttribute($attribute)
    {
        return Attribute::updateOrCreate(['code' => $attribute], [
            'label' => ucwords(snake_case($attribute, '')),
            'code' => $attribute,
            'type' => 'select'
        ]);
    }
}
