<?php

namespace Ignite\Catalog\Http\Forms;

use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\Item;
use Kris\LaravelFormBuilder\Form;

class ItemForm extends Form
{
    /**
     * Build the form.
     *
     * @return self
     */
    public function buildForm()
    {
        $this->add('type', 'hidden', $this->getTypeOptions());
        $this->add('name', 'text', $this->getNameOptions());
        $this->add('sku', 'text', $this->getSkuOptions());
        $this->add('catalog_id', 'select', $this->getCatalogOptions());
        $this->add('catalog_vendor_id', 'select', $this->getVendorOptions());
        $this->add('class', 'select', $this->getClassOptions());
        $this->add('manufacturer', 'text', $this->getManufacturerOptions());
        $this->add('short_description', 'textarea', $this->getShortDescriptionOptions());
        $this->add('description', 'textarea', $this->getDescriptionOptions());
        $this->add('terms', 'textarea', $this->getTermsOptions());

        // Price/Points
        $this->add('msrp', $this->isSimple() ? 'text' : 'hidden', $this->getMsrpOptions());
        $this->add('cost', $this->isSimple() ? 'text' : 'hidden', $this->getCostOptions());
        $this->add('price', $this->isSimple() ? 'text' : 'hidden', $this->getPriceOptions());
        $this->add('price_margin', $this->isSimple() ? 'slider' : 'hidden', $this->getPriceMarginOptions());
        $this->add('price_markup', $this->isSimple() ? 'slider' : 'hidden', $this->getPriceMarkupOptions());
        $this->add('point_value', $this->isSimple() ? 'number' : 'hidden', $this->getPointValueOptions());
        $this->add('points', $this->isSimple() ? 'text' : 'hidden', $this->getPointsOptions());

        // Categories
        $this->add('categories', 'choice', $this->getCategoriesOptions());

        // Images
        $this->add('image', 'file', $this->getImageOptions());

        if ($this->getModel() instanceof Item) {
            $this->addVendorMeta();
        }

        // Switches
        $this->add('visibility', 'toggle', $this->getVisibilityOptions());
        $this->add('active', 'toggle', $this->getActiveOptions());
        $this->add('vendor_active', 'toggle', $this->getVendorActiveOptions());

        // Attributes
        if ($this->isConfigurable()) {
            $this->add('attributes', 'hidden', [
                'rules' => 'required',
                'attr' => ['id' => 'catalog-attributes-value']
            ]);
        }

        // Buttons
        $this->add('submit', 'submit', $this->getSaveButtonOptions());

        return $this;
    }

    /**
     * The type of item.
     *
     * @return string
     */
    protected function getType()
    {
        return $this->getData('type', 'simple');
    }

    /**
     * Is the form type for a simple item.
     *
     * @return bool
     */
    public function isSimple()
    {
        return 'simple' === $this->getType();
    }

    /**
     * Is the form type for a simple item.
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return 'configurable' === $this->getType();
    }

    /**
     * The options for the type field.
     *
     * @return array
     */
    protected function getTypeOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.type.label'),
            'rules' => 'required|in:simple,configurable'
        ];
    }

    /**
     * The options for the name field.
     *
     * @return array
     */
    protected function getNameOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.name.label'),
            'rules' => 'required|min:2|max:255'
        ];
    }

    /**
     * The options for the sku field.
     *
     * @return array
     */
    protected function getSkuOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.sku.label'),
            'rules' => 'required|min:2|max:255',
        ];
    }

    /**
     * The options for the code field.
     *
     * @return array
     */
    protected function getCodeOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.code.label'),
            'rules' => 'required|min:2|max:255'
        ];
    }

    /**
     * The options for the classification field.
     *
     * @return array
     */
    protected function getClassOptions()
    {
        return [
            'attr' => ['class' => 'form-control select2'],
            'label' => trans('Catalog::item.fields.class.label'),
            'rules' => 'required|max:50',
            'choices' => [
                'merchandise' => 'Merchandise',
                'card' => 'Giftcard',
                'egift' => 'eGift',
                'reloadable' => 'Reloadable',
                'prepaid' => 'Prepaid',
            ]
        ];
    }

    /**
     * The options for the categories field.
     *
     * @return array
     */
    protected function getCategoriesOptions()
    {
        return [
            'attr' => ['class' => 'form-control select2'],
            'label' => trans('Catalog::item.fields.categories.label'),
            'rules' => 'required',
            'expanded' => false,
            'multiple' => true,
            'choices' => Category::query()->onlyActive()->get()->keyBy('id')->map(function ($category) {
                return $category->name;
            })->toArray()
        ];
    }

    /**
     * The options for the vendor field.
     *
     * @return array
     */
    protected function getVendorOptions()
    {
        return [
            'attr' => ['class' => 'form-control select2'],
            'label' => trans('Catalog::item.fields.catalog_vendor_id.label'),
            'rules' => 'required|max:255',
            'choices' => app(\Ignite\Catalog\Models\Options\Vendors::class)->toDropdown()
        ];
    }

    /**
     * The options for the catalog options field.
     *
     * @return array
     */
    protected function getCatalogOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.catalog_id.label'),
            'attr' => ['class' => 'form-control select2'],
            'rules' => 'required|max:255',
            'choices' => app(\Ignite\Catalog\Models\Options\Catalog::class)->toDropdown(),
        ];
    }

    /**
     * The options for the short description field.
     *
     * @return array
     */
    protected function getShortDescriptionOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.short_description.label'),
            'rules' => 'max:255',
            'attr' => ['rows' => 3]
        ];
    }

    /**
     * The options for the description field.
     *
     * @return array
     */
    protected function getDescriptionOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.description.label'),
            'rules' => $this->isConfigurable() ? 'required' : ''
        ];
    }

    /**
     * The options for the terms field.
     *
     * @return array
     */
    protected function getTermsOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.terms.label'),
        ];
    }

    /**
     * The options for the manufacturer field.
     *
     * @return array
     */
    protected function getManufacturerOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.manufacturer.label'),
            'rules' => 'required|max:255',
        ];
    }

    /**
     * The options for the save button.
     *
     * @return array
     */
    protected function getSaveButtonOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.save.label'),
            'attr' => ['class' => 'btn btn-primary'],
        ];
    }
    /**
     * The options for the visibility field.
     *
     * @return array
     */
    protected function getVisibilityOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.visibility.label'),
            'help_block' => [
                'text' => 'Visible in the catalog front-end?',
            ],
        ];
    }

    /**
     * The options for the active field.
     *
     * @return array
     */
    protected function getActiveOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.active.label'),
            'help_block' => [
                'text' => 'Active in the catalog front-end? i.e. Can be added to the cart.',
            ],
        ];
    }

    /**
     * The options for the active field.
     *
     * @return array
     */
    protected function getVendorActiveOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.vendor_active.label'),
            'help_block' => [
                'text' => 'Allow the Order API to connect to the vendor?',
            ],
        ];
    }

    /**
     * The options for the image field.
     *
     * @return array
     */
    protected function getImageOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.image.label'),
            'rules' => 'image'
        ];
    }

    /**
     * The options for the msrp field.
     *
     * @return array
     */
    protected function getMsrpOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.msrp.label'),
            'rules' => 'numeric|max:255',
            'default_value' => '0.00',
        ];
    }

    /**
     * The options for the cost field.
     *
     * @return array
     */
    protected function getCostOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.cost.label'),
            'rules' => 'required|numeric',
            'default_value' => '50.00',
        ];
    }

    /**
     * The options for the price field.
     *
     * @return array
     */
    protected function getPriceOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.price.label'),
            'rules' => 'required|numeric',
            'default_value' => '57.50',
        ];
    }

    /**
     * The options for the points field.
     *
     * @return array
     */
    protected function getPointsOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.points.label'),
            'rules' => 'required|numeric',
            'default_value' => '57.50',
            'attr' => ['readonly' => true]
        ];
    }

    /**
     * The options for the price markup field.
     *
     * @return array
     */
    protected function getPriceMarkupOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.price_markup.label'),
            'rules' => 'required|numeric',
            'attr' => [
                'max' => 50,
                'min' => 0,
                'step' => '0.0100'
            ],
            'default_value' => '15.00',
        ];
    }

    /**
     * The options for the price margin field.
     *
     * @return array
     */
    protected function getPriceMarginOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.price_margin.label'),
            'rules' => 'required|numeric',
            'show_changed_default' => true,
            'attr' => [
                'max' => 50,
                'min' => 0,
                'step' => '0.0100'
            ],
            'default_value' => config('catalog.default_price_margin')
        ];
    }

    /**
     * The options for the point value field.
     *
     * @return array
     */
    protected function getPointValueOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.point_value.label'),
            'rules' => 'required|numeric',
            'attr' => ['readonly'],
            'value' => config('catalog.default_point_value'),
        ];
    }

    /**
     * The options for the point value field.
     *
     * @return array
     */
    protected function getPointsMinOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.points_min.label'),
            'rules' => 'required|numeric',
            'default_value' => 0,
        ];
    }

    /**
     * The options for the point value field.
     *
     * @return array
     */
    protected function getPointsMaxOptions()
    {
        return [
            'label' => trans('Catalog::item.fields.points_max.label'),
            'rules' => 'required|numeric',
            'default_value' => 0,
        ];
    }

    /**
     * Add the vendor meta sub form.
     *
     * @return $this
     */
    protected function addVendorMeta()
    {
        $vendor = $this->getModel()->vendor->name;
        $fields = config("Catalog.vendors.$vendor.schema", []);

        /** @var Form $vendorMetaForm */
        $vendorMetaForm = $this->formBuilder->createByArray($fields, [], [
            'model' => $this->getModel()->vendor_meta
        ]);

        if (empty($fields)) {
            $vendorMetaForm->add('help', 'static', [
                'tag' => 'div',
                'label_show' => false,
                'attr' => ['class' => 'form-control-static alert alert-warning'],
                'value' => "Unable to find any meta configuration for {$vendor}."
            ]);
        }

        $this->add('vendor_meta', 'form', [
            'label' => trans('Catalog::item.fields.vendor_meta.label'),
            'class' => $vendorMetaForm
        ]);

        return $this;
    }
}
