<?php

namespace Ignite\Activity\Solutions;

use Facade\IgnitionContracts\RunnableSolution;
use Ignite\Theme\Manager;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Str;

class SchemaFileNotFoundSolution implements RunnableSolution
{
    /**
     * @var string
     */
    protected $type;

    /**
     * @var string
     */
    protected $themePath;

    /**
     * Set the schema type.
     *
     * @param string $type
     *
     * @return $this
     */
    public function setType($type)
    {
        $this->type = $type;

        return $this;
    }

    /**
     * The solution title should be succinct.
     *
     * @return string
     */
    public function getSolutionTitle(): string
    {
        return 'Missing schema file';
    }

    /**
     * The solution description should include as much detail as possible.
     *
     * @return string
     */
    public function getSolutionDescription(): string
    {
        return "We have detected that the schema file for Activity type {$this->type} is missing. " .
            "A schema json file should be present in your active theme under: /forms/{$this->type}.json";
    }

    /**
     * Provide documentation links to as many sources as possible.
     *
     * @return array
     */
    public function getDocumentationLinks(): array
    {
        return [
            'Ignite Documentation' => 'https://docs.igniteincentives.io',
        ];
    }

    /**
     * The action description should describe what the action will do.
     *
     * @return string
     */
    public function getSolutionActionDescription(): string
    {
        $path = "{$this->getThemePath()}/forms/{$this->type}.json";

        return "Would you like to generate the schema file at the following path? {$path}";
    }

    /**
     * The run button text will provide actionable instruction.
     *
     * @return string
     */
    public function getRunButtonText(): string
    {
        return "Generate schema file for {$this->type}";
    }

    /**
     * The run parameters will be passed to the run method which can in turn be passed to the Artisan command.
     *
     * @return array
     */
    public function getRunParameters(): array
    {
        $file = DIRECTORY_SEPARATOR . 'forms' . DIRECTORY_SEPARATOR . Str::slug($this->type) . '.json';
        return [
            'name' => Str::studly($this->type),
            'path' => $this->getThemePath() . $file,
        ];
    }

    /**
     * The run method will run the artisan command.
     *
     * @param array $parameters
     */
    public function run(array $parameters = [])
    {
        Artisan::call('ignite:activity:make-schema', $parameters);
    }

    /**
     * The path to the currently active theme.
     *
     * @return string
     */
    protected function getThemePath()
    {
        if (! $this->themePath) {
            $this->themePath = app(Manager::class)->current()->path();
        }

        return $this->themePath;
    }
}
