<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Resource;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ResourceTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_casts_data_from_json_to_array()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
            'data' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $resource->data['foo']);
    }

    /**
     * @test
     */
    public function it_casts_offer_data_from_json_to_array()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
            'offer_data' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $resource->offer_data['foo']);
    }

    /**
     * @test
     */
    public function it_casts_rule_data_from_json_to_array()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
            'rule_data' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $resource->rule_data['foo']);
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_type_model()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
        ]);

        $this->assertInstanceOf(Type::class, $resource->type);
        $this->assertTrue($resource->type->is($type));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_offer_model()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
            'offer_id' => $offer->getKey()
        ]);

        $this->assertInstanceOf(Offer::class, $resource->offer);
        $this->assertTrue($resource->offer->is($offer));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_submitter_user_model()
    {
        /** @var User $user */
        $user = factory(User::class)->create();
        /** @var Resource $resource */
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
            'user_id' => $user->getKey()
        ]);

        $this->assertInstanceOf(User::class, $resource->user);
        $this->assertTrue($resource->user->is($user));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_submitter_participant_model()
    {
        /** @var User $user */
        $user = factory(User::class)->create();
        /** @var Participant $participant */
        $participant = factory(Participant::class)->create(['user_id' => $user->getKey()]);

        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
            'user_id' => $user->getKey()
        ]);

        $this->assertInstanceOf(Participant::class, $resource->participant);
        $this->assertTrue($resource->participant->is($participant));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_activity_model()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'submission_id' => $submission->getKey(),
            'activity_id' => $activity->getKey()
        ]);

        $this->assertInstanceOf(Activity::class, $resource->activity);
        $this->assertTrue($resource->activity->is($activity));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_submission_model()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
        ]);

        $this->assertInstanceOf(Submission::class, $resource->submission);
        $this->assertTrue($resource->submission->is($submission));
    }
}
