<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Domain\Offers\OfferFormFactory;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\Type;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Kris\LaravelFormBuilder\Form;

class OfferTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_provides_a_method_to_get_the_label()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create([
            'label' => 'My first offer',
        ]);

        $this->assertEquals('My first offer', $offer->label());
    }

    /**
     * @test
     */
    public function it_provides_a_method_to_get_the_code()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create([
            'label' => 'My first offer',
            'code' => 'my-first-offer',
        ]);

        $this->assertEquals('my-first-offer', $offer->code());
    }

    /**
     * @test
     */
    public function it_automatically_converts_the_name_value_to_code_slug_when_code_slug_is_empty()
    {
        $offer = factory(Offer::class)->create([
            'label' => 'My first offer',
            'code' => ''
        ]);

        $this->assertEquals('my-first-offer', $offer->code);
    }

    /**
     * @test
     */
    public function it_provides_the_front_url_for_the_offer()
    {
        $type = factory(Type::class)->create([
            'code' => 'sales'
        ]);

        $offer = factory(Offer::class)->create([
            'type_id' => $type->getKey(),
            'code' => '2021-q1-sales'
        ]);

        $this->assertEquals(url('/activity/sales/2021-q1-sales'), $offer->url());
    }

    /**
     * @test
     */
    public function it_can_determine_whether_the_status_is_active()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['status' => Offer::ACTIVE]);

        $this->assertTrue($offer->isActive());
    }

    /**
     * @test
     */
    public function it_can_determine_whether_the_status_is_inactive()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['status' => Offer::INACTIVE]);

        $this->assertTrue($offer->isInactive());
    }

    /**
     * @test
     */
    public function it_can_get_the_status_as_an_array()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create();

        $this->assertIsArray($statuses = $offer->getStatuses());
        $this->assertArrayHasKey(Offer::INACTIVE, $statuses);
        $this->assertArrayHasKey(Offer::ACTIVE, $statuses);
        $this->assertEquals('Inactive', $statuses[Offer::INACTIVE]);
        $this->assertEquals('Active', $statuses[Offer::ACTIVE]);
    }

    /**
     * @test
     */
    public function it_can_determine_whether_it_can_earn_points()
    {
        $this->assertTrue(factory(Offer::class)->create(['is_redeemable' => 1])->isRedeemable());
        $this->assertFalse(factory(Offer::class)->create(['is_redeemable' => 0])->isRedeemable());
    }

    /**
     * @test
     */
    public function it_casts_meta_from_json_to_array()
    {
        $offer = factory(Offer::class)->create([
            'meta' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $offer->meta['foo']);
    }

    /**
     * @test
     */
    public function it_casts_translations_from_json_to_array()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create([
            'translations' => [
                'en' => ['foo' => 'bar']
            ]
        ]);

        $this->assertEquals(['foo' => 'bar'], $offer->translations['en']);
    }

    /**
     * @test
     */
    public function it_provides_an_easy_way_to_get_translations()
    {
        $offer = factory(Offer::class)->create([
            'translations' => [
                'en' => ['foo' => 'bar']
            ]
        ]);

        $this->assertEquals('bar', $offer->translation('foo'));
    }

    /**
     * @test
     */
    public function it_provides_an_easy_way_to_set_translations()
    {
        $offer = factory(Offer::class)->create([
            'translations' => [
                'en' => ['foo' => 'bar']
            ]
        ]);

        $offer->translationSet('foo', 'changed');

        $this->assertEquals('changed', $offer->translation('foo'));
    }

    /**
     * @test
     */
    public function it_provides_an_easy_way_to_save_translations()
    {
        $offer = factory(Offer::class)->create([
            'translations' => [
                'en' => ['foo' => 'bar']
            ]
        ]);

        $offer->translationSave('foo', 'changed');

        $offer = Offer::find($offer->getKey());

        $this->assertEquals('changed', $offer->translation('foo'));
    }

    /**
     * @test
     */
    public function it_can_get_the_offer_form_instance_an_pass_options_and_data()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();

        $offerForm = app(OfferFormFactory::class);
        $offerForm->register($type->offer->related_name, OfferTestFormInstance::class);

//        app()->singleton($type->related_name, function () use ($type) {
//            return new Schema($this->getSchemaData());
//        });

        /** @var Form $instance */
        $instance = $type->offer->form(['method' => 'POST'], ['foo' => 'bar']);

        $this->assertInstanceOf(OfferTestFormInstance::class, $instance);
        $this->assertArrayHasKey('method', $instance->getFormOptions());
        $this->assertArrayHasKey('foo', $instance->getData());
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_type_model()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);

        $this->assertInstanceOf(Type::class, $offer->type);
        $this->assertTrue($offer->type->is($type));
    }

    /**
     * @test
     */
    public function it_defines_a_has_many_relationship_to_the_activity_model()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);

        $this->assertInstanceOf(Collection::class, $offer->activities);
        $this->assertCount(1, $offer->activities);
        $this->assertTrue($offer->activities->contains($activity->getKeyName(), '=', $activity->getKey()));
    }

    /**
     * @test
     */
    public function it_defines_a_many_to_many_relationship_to_the_rules()
    {
        /** @var Rule $rule */
        $rules = factory(Rule::class, 2)->create();

        /** @var Offer $offer */
        $offer = factory(Offer::class)->create();

        foreach ($rules as $rule) {
            $offer->rules()->save($rule);
        }

        $this->assertCount(2, $offer->rules);
    }

    /**
     * @test
     */
    public function it_can_scope_a_query_to_sort_by_sequence()
    {
        $offer1 = factory(Offer::class)->create(['sequence' => null]);
        $offer2 = factory(Offer::class)->create(['sequence' => 1]);
        $offer3 = factory(Offer::class)->create(['sequence' => 2]);

        $offers = Offer::query()->sequential()->get();

        $this->assertTrue($offers->first()->is($offer2));
    }
}

class OfferTestFormInstance extends Form implements \Ignite\Activity\Contracts\OfferForm
{

}
