<?php

namespace Ignite\Activity\Tests;

use Facade\Ignition\IgnitionServiceProvider;
use Ignite\Activity\Contracts\OfferForm;
use Ignite\Activity\Domain\Schema\Schema;
use Ignite\Activity\Domain\Offers\OfferFormFactory;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Type;
use Ignite\Activity\Providers\ActivityServiceProvider;
use Ignite\Core\Entities\Group;
use Ignite\Core\Entities\User;
use Ignite\Core\Facades\Program;
use Ignite\Core\Providers\CoreServiceProvider;
use Ignite\Flash\Facades\Flash;
use Ignite\Flash\FlashServiceProvider;
use Ignite\Theme\Manager;
use Ignite\Theme\Theme;
use Ignite\Theme\ThemeServiceProvider;
use Mockery;
use Nwidart\Modules\LaravelModulesServiceProvider;
use Spatie\Menu\Laravel\MenuServiceProvider;
use Yajra\DataTables\DataTablesServiceProvider;
use Yajra\DataTables\HtmlServiceProvider;

class TestCase extends \Orchestra\Testbench\TestCase
{
    /**
     * Define the package providers.
     *
     * @param \Illuminate\Foundation\Application $app
     *
     * @return array
     */
    protected function getPackageProviders($app)
    {
        parent::getPackageProviders($app);

        $app->config->set('modules.scan.enabled', true);
        $app->config->set('modules.scan.paths', [realpath(__DIR__ . '/../../')]);

        return [
            LaravelModulesServiceProvider::class,
            IgnitionServiceProvider::class,
            DataTablesServiceProvider::class,
            HtmlServiceProvider::class,
            FlashServiceProvider::class,
            MenuServiceProvider::class,
            CoreServiceProvider::class,
            ActivityServiceProvider::class,
        ];
    }

    /**
     * Get package aliases.
     *
     * @param  \Illuminate\Foundation\Application  $app
     *
     * @return array
     */
    protected function getPackageAliases($app)
    {
        return [
            'Validator' => \Illuminate\Support\Facades\Validator::class,
            'Form' => \Collective\Html\FormFacade::class,
            'Html' => \Collective\Html\HtmlFacade::class,
            'Flash' => Flash::class,
            'Program' => Program::class,
        ];
    }

    /**
     * Define environment setup.
     *
     * @param  \Illuminate\Foundation\Application  $app
     * @return void
     */
    protected function getEnvironmentSetUp($app)
    {
        /** @var Manager $theme */
        $theme = $app->make(Manager::class);
        $theme->register(new Theme('test', __DIR__ . '/stubs/themes/test'), true);
        $app->instance(Manager::class, $theme);

        $app['config']->set('activity.schema', [
            'claims' => 'forms/claims.json'
        ]);

        $app['config']->set('app.env', 'testing');
        $app['config']->set('database.default', 'testing');
        $app['config']->set('database.connections.testing', [
            'driver' => 'mysql',
            'host' => '127.0.0.1',
            'port' => '3306',
            'database' => getenv('MYSQL_DATABASE') ?: 'ignite_tests',
            'username' => getenv('MYSQL_USERNAME') ?: 'homestead',
            'password' => getenv('MYSQL_PASSWORD') ?: 'secret',
        ]);
    }

    /**
     * Login as admin with groups and permissions applied.
     *
     * @param int|User $identifier
     *
     * @return User
     */
    protected function loginAsAdmin($identifier = 1)
    {
        /** @var User $user */
        if (! $identifier instanceof User) {
            try {
                $user = User::with(['groups'])->where('user_id', $identifier)->firstOrFail();
            } catch (\Exception $e) {
                $user = factory(User::class)->create();
                $group = factory(Group::class)->create();
                $user->groups()->save($group);
            }
        }

        if (! $user->relationLoaded('groups')) {
            $user->fresh()->load('groups');
        }

        $this->app['authorization']->forgetPermissions()->registerPermissions();

        $this->actingAs($user);

        return $user;
    }

    /**
     * Build all the dependencies for an Activity Type.
     *
     * @param array $schema
     *
     * @return \Illuminate\Database\Eloquent\Collection|\Illuminate\Database\Eloquent\Model|mixed
     * @throws \Ignite\Theme\Exception\ThemeNotFoundException
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     */
    protected function buildActivityType($schema = [])
    {
        $type = factory(Type::class)->create([
            'label' => 'Claims',
            'code' => 'claims'
        ]);

        $offer = factory(Offer::class)->create([
            'type_id' => $type->getKey(),
            'sequence' => 1
        ]);

        $type->setRelation('offer', $offer);

        return $type;
    }

    /**
     * Basic schema data.
     *
     * @param array $overrides
     *
     * @return array
     */
    protected function getSchemaData($overrides = [])
    {
        return array_merge([
            'model' => '',
            'statuses' => [
                'states' => [
                    'pending' => [
                        "name" => "pending",
                        "type" => "initial",
                        "color" => "yellow",
                        "icon" => "question"
                    ],
                    'approved' => [
                        "name" => "approved",
                        "type" => "final",
                        "color" => "green",
                        "icon" => "tick"
                    ]
                ],
                "transitions" => [
                    "pending" => [
                        "from" => ["pending"],
                        "to" => "pending"
                    ],
                    "approve" => [
                        "from" => ["pending"],
                        "to" => "approved"
                    ]
                ]
            ],
            'fields' => []
        ], $overrides);
    }
}
