<?php

namespace Ignite\Activity\Http\Controllers;

use Exception;
use Ignite\Activity\Contracts\ActivitySubmissionRepository;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Type;
use Ignite\Activity\Traits\TypeStrategyClientTrait;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Form\Form;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class ActivityTypeController extends Controller
{
    use TypeStrategyClientTrait;

    /**
     * Display a listing of the resource.
     *
     * @param Type $type
     * @param Request $request
     *
     * @return Renderable|RedirectResponse|JsonResponse
     */
    public function index(Type $type, Request $request)
    {
        /** var TypeStrategyInterface */
        $strategy = $this->getTypeStrategy($type->code);
        if ($strategy && ($url = $strategy->activityTypeIndexRedirect($request))) {
            return redirect($url);
        }

        if ($type->hasOneOffer()) {
            return redirect($type->getFirstOffer()->url());
        }

        return view()->first([
            "activity.type.{$type->code}.index",
            'activity.type.index'
        ], compact('type'));
    }

    /**
     * Create the activity type offer form submission.
     *
     * @param Type $type
     * @param Offer $offer
     *
     * @return Renderable
     */
    public function create(Type $type, Offer $offer)
    {
        /** var TypeStrategyInterface */
        $strategy = $this->getTypeStrategy($type->code);
        if ($strategy && ($url = $strategy->activityTypeCreateRedirect($offer, request()))) {
            return redirect($url);
        }

        return view()->first([
            "activity.type.{$type->code}.{$offer->code}.create",
            "activity.type.{$type->code}.create",
            'activity.type.create'
        ], compact('type', 'offer'));
    }

    /**
     * Store the form data.
     *
     * @param Type $type
     * @param Offer $offer
     * @param ActivitySubmissionRepository $submissionRepository
     *
     * @return RedirectResponse
     * @throws Exception
     */
    public function store(Type $type, Offer $offer, ActivitySubmissionRepository $submissionRepository)
    {
        if (! $offer->relationLoaded('type')) {
            $offer->setRelation('type', $type);
        }

        /** @var Form $form */
        $form = $offer->form();

        $form->redirectIfNotValid();

        try {
            //event(new ActivitySubmitting($type, $offer, $form));

            $activity = $submissionRepository->create($offer, $form->getFieldValues(), auth()->user());

            //event(new ActivitySubmitted($type, $offer, $activity));

            return redirect()->route('activity.show', compact('type', 'offer', 'activity'));
        } catch (Exception $exception) {
            logger()->error($exception->getMessage(), compact('exception'));

            $this->flashError('An error occurred while trying to process your submission.');

            return redirect()->back();
        }
    }

    /**
     * Show the offer.
     *
     * @param Type $type
     * @param Offer $offer
     * @param Activity $activity
     *
     * @return Renderable
     */
    public function show(Type $type, Offer $offer, Activity $activity)
    {
        // TODO: Replace with a policy?
        if (auth()->id() !== $activity->submission->user_id) {
            abort(403);
        }

        return view()->first([
            "activity.type.{$type->code}.{$offer->code}.show",
            "activity.type.{$type->code}.show",
            'activity.type.show'
        ], compact('type', 'offer', 'activity'));
    }
}
