<?php

namespace Ignite\Activity\Generators;

use Exception;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Str;

abstract class Generator
{
    /**
     * @var Filesystem
     */
    protected $filesystem;

    /**
     * Generator.
     *
     * @param Filesystem $filesystem
     */
    public function __construct(Filesystem $filesystem)
    {
        $this->filesystem = $filesystem;
    }

    /**
     * Generate the stub file and place it in the given directory.
     *
     * @param string $path
     * @param array $vars
     *
     * @return string
     */
    abstract public function generate(string $path, array $vars = []);

    /**
     * Get the stub file.
     *
     * @param string $path
     *
     * @return string
     * @throws \Illuminate\Contracts\Filesystem\FileNotFoundException
     */
    protected function stub($path)
    {
        //TODO: Replace with a check to exists first and throw a custom exception
        return $this->filesystem->get($path);
    }

    /**
     * Interpolate the variables into the stub.
     *
     * @param string $stub
     * @param array $vars
     *
     * @return string
     */
    protected function interpolate($stub, $vars = [])
    {
        foreach ($vars as $var => $value) {
            $stub = str_replace('Dummy' . ucfirst($var), $value, $stub);
        }

        return $stub;
    }

    /**
     * Store the contents at the given path and return the path.
     *
     * @param string $path
     * @param string $contents
     *
     * @return string
     */
    protected function store($path, $contents)
    {
        $this->forceMakeRecursiveDirectory($path);

        $this->filesystem->put($path, $contents);

        return $path;
    }

    /**
     * Force the creation of a directory, recursively if necessary.
     *
     * @param string $path
     * @param int $mode
     * @return bool
     */
    protected function forceMakeRecursiveDirectory($path, $mode = 0755)
    {
        return $this->filesystem->makeDirectory(dirname($path), $mode, true, true);
    }

    /**
     * Throw an exception when the given placeholder key is not in the variables array.
     *
     * @param array $vars
     * @param string $key
     *
     * @throws Exception
     */
    protected function guardAgainstMissingPlaceholder($vars, $key)
    {
        foreach ($vars as $var => $value) {
            $vars[Str::slug($var, '_')] = $value;
        }

        if (! array_key_exists(Str::slug($key, '_'), $vars)) {
            throw new Exception("Placeholder {$key} is missing from " . json_encode($vars));
        }
    }
}
