<?php

namespace Ignite\Activity\Entities;

use Ignite\Activity\Facades\Schema;
use Ignite\Core\Entities\Base;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\Scopes\ParticipantScope;
use Ignite\Core\Entities\User;
use Illuminate\Contracts\Auth\Access\Authorizable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;

/**
 * Activity Entity
 *
 * @property int $id
 * @property int $submitted_by_user_id
 * @property int $offer_id
 * @property array $meta
 * @property User $user
 * @property Participant $participant
 * @property Offer $offer
 * @property Type $type
 * @property Submission $submission
 * @property Resource $resource
 * @property Model $data
 * @property Carbon $created_at
 * @property Carbon $updated_at
 * @property int|null $type_id
 * @property-read \Illuminate\Database\Eloquent\Collection|Submission[] $submissions
 * @property-read int|null $submissions_count
 * @method static \Illuminate\Database\Eloquent\Builder|Activity newModelQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|Activity newQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|Activity query()
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereCreatedAt($value)
 * @method static Builder|Base whereHasPermission(string $permission, ?\Ignite\Core\Entities\User $user = null)
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereMeta($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereOfferId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereSubmittedByUserId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereTypeId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Activity whereUpdatedAt($value)
 * @mixin \Eloquent
 */
class Activity extends Base
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'activity';

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'meta' => 'json',
    ];

    /**
     * @var string
     */
    protected static $defaultType = null;

    /**
     * Set the default Activity Type for $this->data() for when we do not have a model.
     * Usually defaults to the first one.
     *
     * @param string $type
     */
    public static function setDefaultType(string $type): void
    {
        static::$defaultType = $type;
    }

    /**
     * The relationship to the submitter's user record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'submitted_by_user_id', 'user_id');
    }

    /**
     * The relationship to the submitter's user record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function participant()
    {
        return $this->belongsTo(Participant::class, 'submitted_by_user_id', 'user_id')
            ->withoutGlobalScope(app(ParticipantScope::class));
    }

    /**
     * The relationship to the offer record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function offer()
    {
        return $this->belongsTo(Offer::class, 'offer_id', 'id');
    }

    /**
     * The relationship to the offer record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOneThrough
     */
    public function type()
    {
        return $this->hasOneThrough(Type::class, Offer::class, 'id', 'id', 'offer_id', 'type_id');
    }

    /**
     * The relationship to the submission record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function submission()
    {
        return $this->hasOne(Submission::class, 'activity_id', 'id')->where('user_id', $this->submitted_by_user_id);
    }

    /**
     * The relationship to the submission records.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function submissions()
    {
        return $this->hasMany(Submission::class, 'activity_id', 'id');
    }

    /**
     * The relationship to the submission record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function resource()
    {
        return $this->belongsTo(Resource::class, 'id', 'activity_id');
    }

    /**
     * The relationship to the data record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function data()
    {
        if ($this->id || empty(static::$defaultType)) {
            // if we know what type the model is
            // or just default to the first one
            $model = Schema::type($this->type->code)->model();
        } else {
            // if we do not have a model, but have a designated default type
            $model = Schema::type(static::$defaultType)->model();
        }

        return $this->hasOne($model, 'activity_id', 'id');
    }

    /**
     * A convenience method for checking whether the given user has the ability to perform an action this activity.
     *
     * @param string $action
     * @param Authorizable|null $user
     * @return bool
     */
    public function can(string $action, ?Authorizable $user = null): bool
    {
        return $this->type->can($action, $user);
    }
}
