<?php

namespace Ignite\Activity\Domain\Schema;

use Exception;
use Ignite\Activity\Domain\Offers\OfferFormFactory;
use Ignite\Activity\Domain\Rules\RuleFactory;
use Ignite\Theme\Manager as ThemeManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Container\Container;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Filesystem\Filesystem;

class Manager
{
    /**
     * @var Container
     */
    private $container;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @var ThemeManager
     */
    private $themeManager;

    /**
     * Manager constructor.
     *
     * @param Container $container
     *
     * @throws BindingResolutionException
     */
    public function __construct(Container $container)
    {
        $this->container = $container;
        $this->filesystem = $container->make(Filesystem::class);
        $this->themeManager = $container->make(ThemeManager::class);
    }

    /**
     * Get an instance of the Schema by finding the schema file with the given code.
     *
     * @param string $file
     *
     * @return Schema
     * @throws FileNotFoundException
     * @throws SchemaParseException
     */
    public function createFromFile($file)
    {
        $activeTheme = $this->themeManager->current();

        $json = $this->filesystem->get($activeTheme->path() . DIRECTORY_SEPARATOR . $file);

        $data = json_decode($json, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new SchemaParseException($file, json_last_error_msg());
        }

        return new Schema($data);
    }

    /**
     * Resolve the schema for an activity type.
     *
     * @param string $code
     *
     * @return Schema
     * @throws SchemaNotFoundException|SchemaNotConfiguredException
     */
    public function type($code)
    {
        try {
            return $this->container->get($code);
        } catch (Exception $exception) {
            if (! array_key_exists($code, $this->container->config->get('activity.schema', []))) {
                throw new SchemaNotConfiguredException($code);
            } else {
                throw new SchemaNotFoundException($code);
            }
        }
    }
}
