<?php

namespace Ignite\Activity\Domain\Rules;

use Exception;
use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Entities\Rule;

class RuleFactory
{
    /**
     * The registered Rule classes.
     *
     * @var array
     */
    protected $rules = [];

    /**
     * RuleFactory instance.
     *
     * @param array $rules
     */
    public function __construct(array $rules = [])
    {
        $this->rules = $rules;
    }

    /**
     * Register a Rule class using a short-name as the key.
     *
     * @param string $name
     * @param string $class
     *
     * @return $this
     */
    public function register($name, $class)
    {
        $this->rules[$name] = $class;

        return $this;
    }

    /**
     * Deregister a Rule class using a short-name as the key.
     *
     * @param string $name
     *
     * @return $this
     */
    public function deregister($name)
    {
        unset($this->rules[$name]);

        return $this;
    }

    /**
     * Check if the given name and/or class has already been registered.
     *
     * @param string $name
     * @param string|null $class
     * @return false
     */
    public function has(string $name, ?string $class = null)
    {
        if (array_key_exists($name, $this->rules) && is_null($class)) {
            return true;
        }

        if (array_key_exists($name, $this->rules) && $class === $this->rules[$name]) {
            return true;
        }

        return false;
    }

    /**
     * Make an instance of a Rule class given name.
     *
     * @param string $name
     * @param array $params
     *
     * @return RuleContract
     * @throws Exception
     */
    public function make($name, array $params = [])
    {
        if (! $this->has($name)) {
            throw RuleException::notFound($name);
        }

        if (! isset($params['rule']) || ! $params['rule'] instanceof Rule) {
            throw RuleException::incorrectParameter($name, $params);
        }

        $instance = app($this->rules[$name], $params);

        if (! $instance instanceof RuleContract) {
            throw RuleException::missingInterface($instance, $name);
        }

        return $instance;
    }
}
