<?php

namespace Ignite\Activity\Domain\Notifications;

use Exception;
use Illuminate\View\Compilers\BladeCompiler;
use Illuminate\View\Factory;

class Template
{
    /**
     * @var BladeCompiler
     */
    private $bladeCompiler;

    /**
     * @var Factory
     */
    private $viewFactory;

    /**
     * Template instance.
     *
     * @param BladeCompiler $bladeCompiler
     * @param Factory $viewFactory
     */
    public function __construct(BladeCompiler $bladeCompiler, Factory $viewFactory)
    {
        $this->bladeCompiler = $bladeCompiler;
        $this->viewFactory = $viewFactory;
    }

    /**
     * Compile the template string.
     *
     * @param string $value
     * @param array $args
     *
     * @return string
     * @throws TemplateException
     */
    public function compile($value, array $args = [])
    {
        $value = $this->strip($value, $args);

        $compiled = $this->bladeCompiler->compileString($value);

        $args['__env'] = $this->viewFactory;

        ob_start() and extract($args, EXTR_SKIP);

        try {
            eval('?' . '>' . $compiled);
        } catch (Exception $e) {
            ob_get_clean();
            throw new TemplateException($e->getMessage(), null, $e);
        }

        $content = ob_get_clean();

        return $content ?? '';
    }

    /**
     * Strip out anything malicious in the template.
     *
     * @param string $value
     * @param array $args
     *
     * @return string
     */
    protected function strip($value, array $args = [])
    {
        $value = $this->stripPhpTags($value);
        $value = $this->stripPhpDirectives($value);
        $value = $this->stripCsrfToken($value);
        $value = $this->stripMethodDirective($value);
        $value = $this->stripErrorDirective($value);
        $value = $this->stripJsonDirective($value);
        $value = $this->stripIncludeDirective($value);
        $value = $this->stripExtendsDirective($value);
        $value = $this->stripSectionDirective($value);
        $value = $this->stripYieldDirective($value);
        $value = $this->stripComponentDirective($value);
        $value = $this->stripWhileDirective($value);
        $value = $this->stripAuthDirective($value);
        $value = $this->stripGuestDirective($value);
        $value = $this->stripStackDirective($value);
        $value = $this->stripInjectDirective($value);

        return trim($value);
    }

    /**
     * Strip out any PHP tags.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripPhpTags($value)
    {
        return preg_replace('/(<\?php)(.*)(\?>)?/', '', $value);
    }

    /**
     * Strip out any PHP tags.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripPhpDirectives($value)
    {
        return preg_replace('/(@php)(.*)(@endphp)?/', '', $value);
    }

    /**
     * Strip out a csrf token.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripCsrfToken($value)
    {
        return preg_replace('/(@csrf)/', '', $value);
    }

    /**
     * Strip out a method directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripMethodDirective($value)
    {
        return preg_replace('/(@method\((.*)?)/', '', $value);
    }

    /**
     * Strip out an error directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripJsonDirective($value)
    {
        return preg_replace('/(@json\((.*)?)/', '', $value);
    }

    /**
     * Strip out an error directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripErrorDirective($value)
    {
        return preg_replace('/(@error\((.*)?)/', '', $value);
    }

    /**
     * Strip out an include directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripIncludeDirective($value)
    {
        $value = preg_replace('/(@include\((.*)?)/', '', $value);
        $value = preg_replace('/(@includeIf\((.*)?)/', '', $value);
        $value = preg_replace('/(@includeWhen\((.*)?)/', '', $value);
        $value = preg_replace('/(@includeUnless\((.*)?)/', '', $value);
        $value = preg_replace('/(@includeFirst\((.*)?)/', '', $value);
        $value = preg_replace('/(@each\((.*)?)/', '', $value);

        return $value;
    }

    /**
     * Strip out a while directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripWhileDirective($value)
    {
        return preg_replace('/(@while)(.*)(@endwhile)/s', '', $value);
    }

    /**
     * Strip out an extends directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripExtendsDirective($value)
    {
        return preg_replace('/(@extends\((.*)?)/', '', $value);
    }

    /**
     * Strip out a while directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripSectionDirective($value)
    {
        $value = preg_replace('/(@section)(.*)(@endsection)/s', '', $value);
        $value = preg_replace('/(@hasSection)(.*)(@endif)/s', '', $value);
        $value = preg_replace('/(@sectionMissing)(.*)(@endif)/s', '', $value);

        return $value;
    }

    /**
     * Strip out an error directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripYieldDirective($value)
    {
        return preg_replace('/(@yield\((.*)?)/', '', $value);
    }

    /**
     * Strip out a while directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripComponentDirective($value)
    {
        return preg_replace('/(@component)(.*)(@endcomponent)/s', '', $value);
    }

    /**
     * Strip out a auth directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripAuthDirective($value)
    {
        return preg_replace('/(@auth)(.*)(@endauth)/s', '', $value);
    }

    /**
     * Strip out a guest directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripGuestDirective($value)
    {
        return preg_replace('/(@guest)(.*)(@endguest)/s', '', $value);
    }

    /**
     * Strip out a while directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripStackDirective($value)
    {
        $value = preg_replace('/(@stack\((.*)?)/', '', $value);
        $value = preg_replace('/(@push)(.*)(@endpush)/s', '', $value);
        $value = preg_replace('/(@prepend)(.*)(@endprepend)/s', '', $value);

        return $value;
    }

    /**
     * Strip out a while directive.
     *
     * @param string $value
     *
     * @return string
     */
    protected function stripInjectDirective($value)
    {
        return preg_replace('/(@inject\((.*)?)/', '', $value);
    }
}
