<?php

namespace Ignite\Activity\Http\Controllers\Admin;

use Ignite\Activity\Contracts\ActivitySubmissionRepository;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Activity\Models\Grid\ActivitySubmissionTable;
use Ignite\Core\Entities\User;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Form\Form;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Throwable;

class SubmissionController extends Controller
{
    /**
     * @var ActivitySubmissionRepository
     */
    protected ActivitySubmissionRepository $submissionRepository;

    /**
     * SubmissionController constructor.
     *
     * @param ActivitySubmissionRepository $submissionRepository
     */
    public function __construct(ActivitySubmissionRepository $submissionRepository)
    {
        $this->submissionRepository = $submissionRepository;
    }

    /**
     * Show the list of all activity submissions for the type and offer.
     *
     * @param Type $type
     * @param Offer $offer
     * @param ActivitySubmissionTable $table
     * @param Request $request
     *
     * @return Renderable|JsonResponse
     */
    public function index(Type $type, Offer $offer, ActivitySubmissionTable $table, Request $request)
    {
        $table->with('type', $type);
        $table->with('offer', $offer);
        $table->with('status', $status = $request->get('status'));

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Activity::admin.submissions.index', compact('type', 'offer', 'status'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param Type $type
     * @param Offer $offer
     *
     * @return Renderable
     */
    public function create(Type $type, Offer $offer)
    {
        return view('Activity::admin.submissions.create', compact('type', 'offer'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Type $type
     * @param Offer $offer
     * @param Request $request
     *
     * @return RedirectResponse
     * @throws Throwable
     */
    public function store(Type $type, Offer $offer, Request $request)
    {
        /** @var Form $form */
        $form = $offer->form([], compact('type'));

        $form->redirectIfNotValid();

        try {
            $submitter = User::query()->find($request->query->get('user_id')) ?? auth()->user();

            $activity = $this->submissionRepository->create($offer, $form->getFieldValues(), $submitter);

            $this->flashSuccess('Submission created successfully.');

            return redirect()->route('admin.activity.edit', [
                'type' => $type->code,
                'offer' => $offer->code,
                'activity' => $activity->getKey(),
                'submission' => $activity->submission->getKey(),
            ]);
        } catch (Throwable $exception) {
            logger()->error($exception->getMessage(), compact('exception'));

            $this->flashError('An error occurred while trying to process your submission. Please contact IT.');

            return redirect()->back();
        }
    }

    /**
     * Show the specified resource.
     *
     * @param Type $type
     * @param Offer $offer
     * @param Activity $activity
     * @param Submission $submission
     *
     * @return Renderable
     */
    public function show(Type $type, Offer $offer, Activity $activity, Submission $submission)
    {
        if (!$this->guardAgainstMismatchedSubmission($activity, $submission)) {
            $this->flashError('The Submission ID is invalid for the provided Activity.');
            return redirect()->route('admin.activity.index', [
                 'type' => $type->code,
                 'offer' => $offer->code,
            ]);
        }

        return view('Activity::admin.submissions.show', compact('type', 'offer', 'activity', 'submission'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Type $type
     * @param Offer $offer
     * @param Activity $activity
     * @param Submission $submission
     *
     * @return Renderable
     */
    public function edit(Type $type, Offer $offer, Activity $activity, Submission $submission)
    {
        if (!$this->guardAgainstMismatchedSubmission($activity, $submission)) {
            $this->flashError('The Submission ID is invalid for the provided Activity.');
            return redirect()->route('admin.activity.index', [
                 'type' => $type->code,
                 'offer' => $offer->code,
            ]);
        }

        return view('Activity::admin.submissions.edit', compact('type', 'offer', 'activity', 'submission'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Type $type
     * @param Offer $offer
     * @param Activity $activity
     * @param Submission $submission
     *
     * @return RedirectResponse
     * @throws \Exception
     */
    public function update(Type $type, Offer $offer, Activity $activity, Submission $submission)
    {
        /** @var Form $form */
        $form = $offer->form(['model' => $activity->data], compact('type', 'activity', 'submission'));

        $form->redirectIfNotValid();

        try {
            $user = auth()->user();

            $submission = $this->submissionRepository->update($submission->getKey(), $form->getFieldValues(), $user);

            $this->flashSuccess('Submission updated successfully.');

            return redirect()->route('admin.activity.edit', [
                'type' => $type->code,
                'offer' => $offer->code,
                'activity' => $activity->getKey(),
                'submission' => $submission->getKey(),
            ]);
        } catch (Throwable $exception) {
            logger()->error($exception->getMessage(), compact('exception'));

            $this->flashError('An error occurred while trying to process your submission. Please contact IT.');

            return redirect()->back();
        }
    }

    /**
     * Remove the specified resource from storage.
     * @param int $id
     * @return Renderable
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Guard that activity and submission matches.
     *
     * @param  Activity   $activity
     * @param  Submission $submission
     * @return bool
     */
    protected function guardAgainstMismatchedSubmission(Activity $activity, Submission $submission): bool
    {
        if ($activity->submission->getKey() !== $submission->getKey()) {
            return false;
        }

        return true;
    }
}
