<?php

namespace Ignite\Activity\Entities;

use Ignite\Activity\Domain\Offers\OfferFormFactory;
use Ignite\Activity\Traits\MetableTrait;
use Ignite\Activity\Traits\SluggableTrait;
use Ignite\Activity\Traits\TranslatableTrait;
use Ignite\Core\Entities\Base;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;

/**
 * Activity Offer Entity
 *
 * @property int $id
 * @property int $type_id
 * @property string $status
 * @property string $label
 * @property string $code
 * @property string $related_name
 * @property bool $is_redeemable
 * @property int $sequence
 * @property array $meta
 * @property array $translations
 * @property Type $type
 * @property Collection|Rule[] $rules
 * @property Collection|Activity[] $activities
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property-read int|null $activities_count
 * @property-read int|null $rules_count
 * @method static Builder|Offer newModelQuery()
 * @method static Builder|Offer newQuery()
 * @method static Builder|Offer onlyActive()
 * @method static Builder|Offer query()
 * @method static Builder|Offer sequential()
 * @method static Builder|Offer whereCode($value)
 * @method static Builder|Offer whereCreatedAt($value)
 * @method static Builder|Base whereHasPermission(string $permission, ?\Ignite\Core\Entities\User $user = null)
 * @method static Builder|Offer whereId($value)
 * @method static Builder|Offer whereIsRedeemable($value)
 * @method static Builder|Offer whereLabel($value)
 * @method static Builder|Offer whereMeta($value)
 * @method static Builder|Offer whereRelatedName($value)
 * @method static Builder|Offer whereSequence($value)
 * @method static Builder|Offer whereStatus($value)
 * @method static Builder|Offer whereTranslations($value)
 * @method static Builder|Offer whereTypeId($value)
 * @method static Builder|Offer whereUpdatedAt($value)
 * @mixin \Eloquent
 */
class Offer extends Base
{
    use SluggableTrait;
    use MetableTrait;
    use TranslatableTrait;

    public const ACTIVE = 1;
    public const INACTIVE = 0;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'activity_offer';

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'is_redeemable' => 'boolean',
        'meta' => 'json',
        'translations' => 'json',
    ];

    /**
     * The sluggable attributes.
     *
     * @var array
     */
    protected $sluggables = [
        'code' => 'label',
    ];

    // Getters

    /**
     * The user friendly label of the Rule.
     *
     * @return string
     */
    public function label()
    {
        return $this->label;
    }

    /**
     * The URL code of the Rule.
     *
     * @return string
     */
    public function code()
    {
        return $this->code;
    }

    /**
     * The url to the offer page on the front-end.
     *
     * @return string
     */
    public function url()
    {
        return route('activity.create', [
            'type' => $this->type->code,
            'offer' => $this->code,
        ]);
    }

    // State

    /**
     * Determine whether the status is active.
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->status === static::ACTIVE;
    }

    /**
     * Determine whether the status is inactive.
     *
     * @return bool
     */
    public function isInactive()
    {
        return $this->status === static::INACTIVE;
    }

    /**
     * The status labels.
     *
     * @return array
     */
    public function getStatuses()
    {
        return [
            static::INACTIVE => 'Inactive',
            static::ACTIVE => 'Active',
        ];
    }

    /**
     * Determine whether the offer can earn points/transactions.
     *
     * @return bool
     */
    public function isRedeemable()
    {
        return $this->is_redeemable;
    }

    // Business Logic

    /**
     * Get the form instance associated with the offer.
     *
     * @param array $options
     * @param array $data
     *
     * @return \Ignite\Activity\Contracts\OfferForm
     * @throws \Exception
     */
    public function form(array $options = [], array $data = [])
    {
        /** @var OfferFormFactory $factory */
        $factory = app(OfferFormFactory::class);

        return $factory->make($this->related_name, array_merge([], $options), array_merge(['offer' => $this], $data));
    }

    // Relationships

    /**
     * The relationship to the type record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function type()
    {
        return $this->belongsTo(Type::class, 'type_id', 'id');
    }

    /**
     * The relationship to the activity records.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function activities()
    {
        return $this->hasMany(Activity::class, 'offer_id', 'id');
    }

    /**
     * The relationship to the collection of rule records.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function rules()
    {
        return $this->belongsToMany(Rule::class, 'activity_offer_rule', 'offer_id', 'rule_id')
            ->withPivot('sequence');
    }

    // Scopes
    // - scopeOnlyRedeemable()
    // - scopeOnlyNonRedeemable()
    // - scopeOnlyActive()
    // - scopeOnlyInactive()
    // - scopeOnlyWithStatus($status)
    // - scopeOnlyWithStatuses([$status])
    // - scopeByLabel($label)
    // - scopeByCode($code)
    // - scopeByType($typeId)
    // - scopeCreatedOn($datetime)
    // - scopeUpdatedOn($datetime)
    // - scopeCreatedBetween($at, $until)
    // - scopeUpdatedBetween($at, $until)

    /**
     * Scope a query to sort records by sequence in ascending order with NULLs last.
     *
     * @param Builder $query
     */
    public function scopeSequential(Builder $query)
    {
        $query->orderByRaw('-sequence DESC');
    }
}
