<?php

namespace Ignite\Activity\Domain\Schema;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use InvalidArgumentException;
use JsonSerializable;

class Fields implements Arrayable, Jsonable, JsonSerializable
{
    /**
     * @var array
     */
    protected $fields;

    /**
     * Fields constructor.
     *
     * @param array $fields
     */
    public function __construct(array $fields)
    {
        $this->hydrate($fields);
    }

    /**
     * Determine whether the given named field exists.
     *
     * @param string $name
     *
     * @return bool
     */
    public function has($name)
    {
        return array_key_exists($name, $this->fields);
    }

    /**
     * Get the field with the given name.
     *
     * @param string $name
     *
     * @return Field
     */
    public function get($name)
    {
        if (! $this->has($name)) {
            throw new InvalidArgumentException("$name field does not exist.");
        }

        return $this->fields[$name];
    }

    /**
     * Add the field to the collection.
     *
     * @param array $field
     *
     * @return self
     */
    public function add(array $field)
    {
        $this->fields[$field['name']] = new Field($field);

        return $this;
    }

    /**
     * Remove the field with the given name.
     *
     * @param string $name
     *
     * @return self
     */
    public function remove($name)
    {
        unset($this->fields[$name]);

        return $this;
    }

    /**
     * Hydrate the fields into field objects keyed by name.
     *
     * @param array $fields
     *
     * @return self
     */
    protected function hydrate(array $fields)
    {
        foreach ($fields as $field) {
            $this->add($field);
        }

        return $this;
    }

    /**
     * Cast the instance to an array.
     *
     * @return array
     */
    public function toArray()
    {
        return $this->fields;
    }

    /**
     * Cast the instance to JSON.
     *
     * @param int $options
     *
     * @return string|void
     */
    public function toJson($options = 0)
    {
        return json_encode($this->toArray(), $options);
    }

    /**
     * Serialize the field as json.
     *
     * @return array
     */
    public function jsonSerialize()
    {
        return $this->toArray();
    }
}
