<?php

namespace Ignite\Activity\Database\Seeders;

use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Database\Seeders\AbstractSeeder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;

/**
 * php artisan db:seed --class="\Ignite\Activity\Database\Seeders\Seeder"
 */
class Seeder extends AbstractSeeder
{
    /**
     * The path to the data file.
     *
     * @var string
     */
    protected $path = '';

    /**
     * Run the seeder.
     *
     * @return void
     */
    public function run()
    {
        parent::run();

        $path = $this->getFilepath('activity.json');
        if (!$path) {
            // if not in config('core.seeder.path folder'), then use default
            $this->path = __DIR__ . '/../../../database/data/';
        }

        $strategy = $this->strategy('json', [
            'filepath' => $this->getFilepath('activity.json')
        ]);

        $sources = $strategy->run();

        foreach ($sources as $source) {
            $this->seed($source);
        }
    }

    /**
     * Seed a single offer.
     *
     * @param array $source
     *
     * @return Type
     */
    protected function seed(array $source)
    {
        // We pull the offers from the type data so we only have data to go into the table columns
        // for `activity_type` left in the $source array. We'll use the offers array later in
        // order to create the relationship records for the new `Type` record.
        $offers = Arr::pull($source['type'], 'offers');

        // Create the activity type in a separate protected method to allow developers to alter the behaviour.
        $type = $this->createActivityType($source);

        if (empty($offers)) {
            return $type;
        }

        // We have created the `Type` instance, we can now associate the offers and, if necessary, the rules.
        return $this->associateOffers($type, $offers);
    }

    /**
     * Create the activity type, isolating the behaviour so that Program seeders can alter if needs be.
     *
     * @param array $source
     *
     * @return Type|Model
     */
    protected function createActivityType(array $source = [])
    {
        $type = Arr::except($source['type'], 'offers');

        return Type::query()->updateOrCreate(
            ['code' => $type['code']],
            $type
        );
    }

    /**
     * Associate the offers for the given type.
     *
     * @param Type $type
     * @param array $offers
     *
     * @return Type
     */
    protected function associateOffers(Type $type, array $offers)
    {
        foreach ($offers as $offer) {
            $this->associateOffer($type, $offer);
        }

        return $type;
    }

    /**
     * Associate an individual offer with the given type.
     *
     * @param Type $type
     * @param array $data
     *
     * @return Offer
     */
    protected function associateOffer(Type $type, array $data)
    {
        $rules = Arr::pull($data, 'rules');

        /** @var Offer $offer */
        $offer = $type->offers()->updateOrCreate(['code' => $data['code']], Arr::except($data, 'rules'));

        if (empty($rules)) {
            return $offer;
        }

        $this->associateRules($offer, $rules);

        return $offer;
    }

    /**
     * Associate the rules for the given offer.

     * @param Offer $offer
     * @param array $rules
     *
     * @return Offer
     */
    protected function associateRules($offer, $rules)
    {
        foreach ($rules as $index => $rule) {
            $this->associateRule($offer, $rule, $index + 1);
        }

        return $offer;
    }

    /**
     * Associate an individual rule with the given offer.
     *
     * @param Offer $offer
     * @param array $data
     * @param int $sequence
     *
     * @return Rule|Model
     */
    protected function associateRule(Offer $offer, array $data, int $sequence)
    {
        $rule = Rule::updateOrCreate(
            [
                'related_name' => $data['related_name'],
                'meta' => \DB::raw("CAST('" . json_encode($data['meta']) . "' AS JSON)")
            ],
            $data
        );

        return $offer->rules()->syncWithoutDetaching([$rule->id => ['sequence' => $sequence]]);
    }
}
