<?php

namespace Ignite\Activity\Tests;

use Exception;
use Ignite\Activity\Domain\Schema\Schema;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Type;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Testing\RefreshDatabase;

class TypeTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_provides_the_type_label_in_singular_and_pluaral_form()
    {
        $type = factory(Type::class)->create([
            'label' => 'Deal Registration'
        ]);

        $this->assertEquals('Deal Registration', $type->singular());
        $this->assertEquals('Deal Registrations', $type->plural());
    }

    /**
     * @test
     */
    public function it_casts_meta_from_json_to_array()
    {
        $type = factory(Type::class)->create([
            'meta' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $type->meta['foo']);
    }

    /**
     * @test
     */
    public function it_can_determine_if_it_has_any_offers_related_without_explicitly_loading_them_first()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        factory(Offer::class, 3)->create(['type_id' => $type->getKey()]);

        $this->assertTrue($type->hasOffers());
    }

    /**
     * @test
     */
    public function it_can_determine_if_it_has_exactly_one_offer()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        factory(Offer::class)->create(['type_id' => $type->getKey()]);

        $this->assertTrue($type->hasOneOffer());
    }

    /**
     * @test
     * @group wip
     */
    public function it_can_get_the_first_offer()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        factory(Offer::class)->create(['type_id' => $type->getKey(), 'sequence' => 5]);
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey(), 'sequence' => 2]);

        $this->assertTrue($type->getFirstOffer()->is($offer));
    }

    /**
     * @test
     */
    public function it_defines_a_has_many_relationship_to_offers()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        /** @var Offer $offer */
        $offer = factory(Offer::class)->create([
            'type_id' => $type->getKey()
        ]);

        $this->assertInstanceOf(Collection::class, $type->offers);
    }

    /**
     * @test
     */
    public function it_defines_a_has_many_relationship_to_the_offers()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        factory(Offer::class)->create(['type_id' => $type->getKey()]);

        $this->assertCount(1, $type->offers);
        $this->assertInstanceOf(Offer::class, $type->offers[0]);
    }

    /**
     * @test
     */
    public function it_will_sort_related_offers_by_sequence_by_default()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        $offer1 = factory(Offer::class)->create(['type_id' => $type->getKey(), 'sequence' => null]);
        $offer2 = factory(Offer::class)->create(['type_id' => $type->getKey(), 'sequence' => 1]);

        $this->assertTrue($type->offers->first()->is($offer2));
    }

    /**
     * @test
     */
    public function it_can_scope_a_query_to_sort_by_sequence()
    {
        $type1 = factory(Type::class)->create(['sequence' => null]);
        $type2 = factory(Type::class)->create(['sequence' => 1]);
        $type3 = factory(Type::class)->create(['sequence' => 2]);

        $types = Type::query()->sequential()->get();

        $this->assertTrue($types->first()->is($type2));
    }

    /**
     * @test
     */
    public function it_can_get_the_schema_for_the_type()
    {
        $type = $this->buildActivityType();

        $this->assertInstanceOf(Schema::class, $type->getSchema());
    }

    /**
     * @test
     */
    public function it_can_get_the_schema_initial_status()
    {
        $type = $this->buildActivityType();

        $this->assertIsArray($type->getInitialStatus());
        $this->assertEquals('pending', $type->getInitialStatus()['name']);
    }

    /**
     * @test
     */
    public function it_can_get_the_schema_initial_status_name()
    {
        $type = $this->buildActivityType();

        $this->assertEquals('pending', $type->getInitialStatusName());
    }

    /**
     * @test
     */
    public function it_will_throw_exception_when_unable_to_find_the_schema_initial_status_name()
    {
        $type = $this->buildActivityType();

        $this->swap('claims', new Schema($this->getSchemaData([
            'statuses' => []
        ])));

        $this->expectException(Exception::class);
        $this->expectExceptionMessage(
            "Unable to find the initial status for activity type: {$type->label}"
        );

        $type->getInitialStatusName();
    }

    /**
     * @test
     */
    public function it_can_get_the_schema_statuses()
    {
        $type = $this->buildActivityType();

        $this->assertIsArray($type->getStatuses());
    }
}
