<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Resource;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\RuleLog;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\SubmissionStatus;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Testing\RefreshDatabase;

class SubmissionTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_defines_a_has_many_relationship_to_the_submission_statuses()
    {
        $type = $this->buildActivityType();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $type->offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        factory(SubmissionStatus::class)->create([
            'submission_id' => $submission->getKey(),
            'label' => 'pending'
        ]);
        factory(SubmissionStatus::class)->create([
            'submission_id' => $submission->getKey(),
            'label' => 'approved'
        ]);
        factory(SubmissionStatus::class)->create([
            'submission_id' => $submission->getKey(),
            'label' => 'cancelled'
        ]);

        $this->assertInstanceOf(Collection::class, $submission->states);
        $this->assertCount(3, $submission->states);
    }

    /**
     * @test
     */
    public function it_defines_a_has_one_relationship_to_the_submission_status()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        factory(SubmissionStatus::class)->create([
            'submission_id' => $submission->getKey(),
            'label' => 'pending',
            'created_at' => '2020-01-01 00:00:00'
        ]);
        factory(SubmissionStatus::class)->create([
            'submission_id' => $submission->getKey(),
            'label' => 'approved',
            'created_at' => '2020-02-01 00:00:00'
        ]);
        $status = factory(SubmissionStatus::class)->create([
            'submission_id' => $submission->getKey(),
            'label' => 'cancelled'
        ]);

        $this->assertInstanceOf(SubmissionStatus::class, $submission->state);
        $this->assertTrue($submission->state->is($status));
    }

    /**
     * @test
     */
    public function it_defines_a_has_one_relationship_to_the_submission_resource()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        $resource = factory(Resource::class)->create([
            'submission_id' => $submission->getKey(),
            'data' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertInstanceOf(Resource::class, $submission->resource);
        $this->assertTrue($submission->resource->is($resource));
    }

    /**
     * @test
     */
    public function it_defines_a_has_one_relationship_to_a_rule_log()
    {
        $rule = factory(Rule::class)->create();
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        $ruleLog = factory(RuleLog::class)->create([
            'submission_id' => $submission->getKey(),
            'details' => [
                'rule_id' => $rule->getKey()
            ]
        ]);

        $this->assertInstanceOf(RuleLog::class, $submission->ruleLog);
        $this->assertTrue($submission->ruleLog->is($ruleLog));
    }

    /**
     * @test
     */
    public function it_defines_a_has_many_relationship_to_the_rule_logs()
    {
        $rule1 = factory(Rule::class)->create();
        $rule2 = factory(Rule::class)->create();

        $type = $this->buildActivityType();

        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        factory(RuleLog::class)->create([
            'submission_id' => $submission->getKey(),
            'details' => [
                'rule_id' => $rule1->getKey()
            ]
        ]);
        factory(RuleLog::class)->create([
            'submission_id' => $submission->getKey(),
            'details' => [
                'rule_id' => $rule2->getKey()
            ]
        ]);

        $this->assertInstanceOf(Collection::class, $submission->ruleLogs);
        $this->assertCount(2, $submission->ruleLogs);
    }

    /**
     * @test
     */
    public function it_defines_a_relationship_to_the_activity()
    {
        $user = factory(User::class)->create();
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
            'user_id' => $user->getKey()
        ]);

        $this->assertInstanceOf(Activity::class, $submission->activity);
        $this->assertTrue($submission->activity->is($activity));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_user_model()
    {
        /** @var User $user */
        $user = factory(User::class)->create();
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
            'user_id' => $user->getKey(),
        ]);

        $this->assertInstanceOf(User::class, $submission->user);
        $this->assertTrue($submission->user->is($user));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_participant_model()
    {
        /** @var User $user */
        $user = factory(User::class)->create();
        /** @var Participant $participant */
        $participant = factory(Participant::class)->create(['user_id' => $user->getKey()]);
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'user_id' => $user->getKey(),
            'activity_id' => $activity->getKey()
        ]);

        $this->assertInstanceOf(Participant::class, $submission->participant);
        $this->assertTrue($submission->participant->is($participant));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_offer_model_through_the_activity()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);

        $this->assertInstanceOf(Offer::class, $submission->offer);
        $this->assertTrue($submission->offer->is($offer));
    }
}
