<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Resource;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ActivityTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_casts_meta_from_json_to_array()
    {
        $activity = factory(Activity::class)->create([
            'meta' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $activity->meta['foo']);
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_submitter_user_model()
    {
        /** @var User $user */
        $user = factory(User::class)->create();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['submitted_by_user_id' => $user->getKey()]);

        $this->assertInstanceOf(User::class, $activity->user);
        $this->assertEquals($user->getKey(), $activity->user->getKey());
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_submitter_participant_model()
    {
        /** @var User $user */
        $user = factory(User::class)->create();
        /** @var Participant $participant */
        $participant = factory(Participant::class)->create(['user_id' => $user->getKey()]);

        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['submitted_by_user_id' => $user->getKey()]);

        $this->assertInstanceOf(Participant::class, $activity->participant);
        $this->assertEquals($participant->getKey(), $activity->participant->getKey());
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_offer_model()
    {
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create();

        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);

        $this->assertInstanceOf(Offer::class, $activity->offer);
        $this->assertEquals($offer->getKey(), $activity->offer->getKey());
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_type_model_through_the_offer()
    {
        /** @var Type $type */
        $type = factory(Type::class)->create();

        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);

        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);

        $this->assertInstanceOf(Type::class, $activity->type);
        $this->assertEquals($type->getKey(), $activity->type->getKey());
    }

    /**
     * @test
     * @group wip
     */
    public function it_defines_a_belongs_to_relationship_to_the_submission_model()
    {
        $type = $this->buildActivityType();
        $user = factory(User::class)->create();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $type->offer->getKey(),
            'submitted_by_user_id' => $user->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
            'user_id' => $user->getKey()
        ]);

        $this->assertInstanceOf(Submission::class, $activity->submission);
        $this->assertTrue($activity->submission->is($submission));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_resource_model()
    {
        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create(['offer_id' => $offer->getKey()]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        /** @var Resource $resource */
        $resource = factory(Resource::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'activity_id' => $activity->getKey(),
            'submission_id' => $submission->getKey(),
        ]);

        $this->assertInstanceOf(Resource::class, $activity->resource);
        $this->assertTrue($activity->resource->is($resource));
    }
}
