<?php

namespace Ignite\Activity\Tests\Domain\Schema;

use Ignite\Activity\Domain\Schema\Field;
use Ignite\Activity\Domain\Schema\Fields;
use Ignite\Activity\Tests\TestCase;
use InvalidArgumentException;

class FieldsTest extends TestCase
{
    /**
     * @test
     */
    public function it_can_cast_to_array()
    {
        $fields = new Fields([['name' => 'foo']]);

        $array = $fields->toArray();

        $item = array_shift($array);

        $this->assertIsArray($array);
        $this->assertArrayHasKey('name', $item->toArray());
        $this->assertEquals('foo', $item->toArray()['name']);
    }

    /**
     * @test
     */
    public function it_can_cast_to_json()
    {
        $fields = new Fields([['name' => 'foo']]);

        $json = $fields->toJson();

        $this->assertIsString($json);
        $this->assertStringContainsString('"name":"foo"', $json);
    }

    /**
     * @test
     */
    public function it_can_cast_to_json_with_options()
    {
        $fields = new Fields([['name' => 'foo']]);

        $json = $fields->toJson(JSON_PRETTY_PRINT);

        $this->assertIsString($json);
        $this->assertStringContainsString('"name": "foo"', $json);
    }

    /**
     * @test
     */
    public function it_can_be_serialized_to_json()
    {
        $fields = new Fields([['name' => 'foo']]);

        $json = json_encode($fields);

        $this->assertStringContainsString('"name":"foo"', $json);
    }

    /**
     * @test
     */
    public function it_can_hydrate_an_array_of_field_data_into_field_objects()
    {
        $fields = [
            ['name' => 'amount'],
            ['name' => 'sold_at']
        ];

        $fields = new Fields($fields);

        $this->assertInstanceOf(Field::class, $fields->get('amount'));
    }

    /**
     * @test
     */
    public function it_can_check_if_it_has_a_field_by_name()
    {
        $fields = [
            ['name' => 'amount'],
            ['name' => 'sold_at']
        ];

        $fields = new Fields($fields);

        $this->assertFalse($fields->has('foo'));
        $this->assertTrue($fields->has('amount'));
    }

    /**
     * @test
     */
    public function it_can_remove_a_field_by_name()
    {
        $fields = [
            ['name' => 'amount'],
            ['name' => 'sold_at']
        ];

        $fields = new Fields($fields);

        $fields->remove('amount');

        $this->assertFalse($fields->has('amount'));
    }

    /**
     * @test
     */
    public function it_can_add_a_field_by_name()
    {
        $fields = [
            ['name' => 'amount']
        ];

        $fields = new Fields($fields);

        $fields->add(['name' => 'sale_id']);

        $this->assertTrue($fields->has('sale_id'));
    }

    /**
     * @test
     */
    public function it_will_throw_an_exception_when_it_cannot_find_a_field_by_name()
    {
        $this->expectException(InvalidArgumentException::class);

        $fields = [
            ['name' => 'amount']
        ];

        $fields = new Fields($fields);

        $fields->get('foo');
    }
}
