<?php

namespace Ignite\Activity\Tests\Unit\Domain\Notifications;

use Ignite\Activity\Domain\Notifications\Template;
use Ignite\Activity\Domain\Notifications\TemplateException;
use Ignite\Activity\Tests\TestCase;

class TemplateTest extends TestCase
{
    /**
     * @var Template
     */
    protected $compiler;
    
    public function setUp() : void
    {
        parent::setUp();

        $this->compiler = $this->app->make(Template::class);
    }
    
    /**
     * @test
     */
    public function it_throws_an_exception_when_it_cannot_compile()
    {
        $this->expectException(TemplateException::class);

        $template = '{{ $var % 1 }}';

        $this->compiler->compile($template, ['var' => 'foo']);
    }

    /**
     * @test
     */
    public function it_can_compile_a_simple_template_string_using_blade()
    {
        $template = '{{ $var }}';

        $compiled = $this->compiler->compile($template, ['var' => 'foo']);

        $this->assertEquals('foo', $compiled);
    }

    /**
     * @test
     */
    public function it_can_compile_an_unescaped_var_using_blade()
    {
        $template = '{!! $var !!}';

        $compiled = $this->compiler->compile($template, ['var' => '<b>foo</b>']);

        $this->assertEquals('<b>foo</b>', $compiled);
    }

    /**
     * @test
     */
    public function it_can_compile_a_for_directive_using_blade()
    {
        $template = <<<'EOD'
@for ($i = 0; $i < 2; $i++)
The current value is {{ $i }}.
@endfor
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("The current value is 0.\nThe current value is 1.", trim($compiled));
    }

    /**
     * @test
     */
    public function it_can_compile_a_foreach_directive_using_blade()
    {
        $template = <<<'EOD'
@foreach ($users as $key => $value)
The value for {{ $key }} is {{ $value }}.
@endforeach
EOD;

        $compiled = $this->compiler->compile($template, [
            'users' => [
                '1' => 'Joe',
                '2' => 'Jane'
            ]
        ]);

        $this->assertEquals("The value for 1 is Joe.\nThe value for 2 is Jane.", trim($compiled));
    }

    /**
     * @test
     */
    public function it_can_compile_a_foreach_continue_directive_using_blade()
    {
        $template = <<<'EOD'
@foreach ($users as $key => $value)
@continue($key == '1')
The value for {{ $key }} is {{ $value }}.
@break($key == '2')
@endforeach
EOD;

        $compiled = $this->compiler->compile($template, [
            'users' => [
                '1' => 'Joe',
                '2' => 'Jane',
                '3' => 'Jack'
            ]
        ]);

        $this->assertEquals("The value for 2 is Jane.", trim($compiled));
    }

    /**
     * @test
     */
    public function it_can_compile_a_forelse_directive_using_blade()
    {
        $template = <<<'EOD'
@forelse ($users as $name)
<li>{{ $name }}</li>
@empty
<p>No users</p>
@endforelse
EOD;

        $compiled = $this->compiler->compile($template, [
            'users' => [
                '1' => 'Joe',
                '2' => 'Jane'
            ]
        ]);

        $this->assertEquals("<li>Joe</li>\n<li>Jane</li>", trim($compiled));

        $compiled = $this->compiler->compile($template, ['users' => []]);

        $this->assertEquals("<p>No users</p>", trim($compiled));
    }

    /**
     * @test
     */
    public function it_can_compile_a_verbatim_directive_using_blade()
    {
        $template = <<<'EOD'
@verbatim
<div class="container">Hello, {{ name }}</div>
@endverbatim
EOD;

        $compiled = $this->compiler->compile($template, ['name' => 'Jane']);

        $this->assertEquals('<div class="container">Hello, {{ name }}</div>', trim($compiled));
    }

    /**
     * @test
     */
    public function it_can_compile_a_switch_directive_using_blade()
    {
        $template = <<<'EOD'
@switch($i)
    @case(1)
        First case...
        @break
    @case(2)
        Second case...
        @break
    @default
        Default case...
@endswitch
EOD;

        $compiled = $this->compiler->compile($template, ['i' => 2]);

        $this->assertEquals('Second case...', trim($compiled));
    }

    /**
     * @test
     */
    public function it_can_compile_an_email_template_string_using_blade()
    {
        $template = <<<'EOD'
# Congratulations {{ $first }}!
@csrf
Your sale submission (#{{ $id }} was received successfully and will be processed shortly.
{{-- This comment will not be present in the rendered HTML --}}
@if ($type === 'sale')
Thank you for selling our product. Keep up the good work!
@endif
EOD;

        $compiled = $this->compiler->compile($template, ['first' => 'Jane', 'id' => 5, 'type' => 'sale']);

        $this->assertEquals(
            "# Congratulations Jane!\n\nYour sale submission (#5 was received successfully and will be processed" .
            " shortly.\n\nThank you for selling our product. Keep up the good work!\n",
            $compiled
        );
    }

    /**
     * @test
     */
    public function it_cannot_compile_php_code_using_blade()
    {
        $template = <<<'EOD'
<?php echo "Hello $first"; ?>
EOD;

        $compiled = $this->compiler->compile($template, ['first' => 'Jane']);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_php_code_using_blade_when_unclosed()
    {
        $template = <<<'EOD'
<?php echo "Hello $first";
EOD;

        $compiled = $this->compiler->compile($template, ['first' => 'Jane']);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_multiple_php_code_using_blade_when_unclosed()
    {
        $template = <<<'EOD'
<?php echo "Hello $first"; ?>
<?php echo "Hello $first";
<?php echo "Hello $first"; ?>
EOD;

        $compiled = $this->compiler->compile($template, ['first' => 'Jane']);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_php_tags_using_blade()
    {
        $template = <<<'EOD'
@php echo "Hello $first"; @endphp
EOD;

        $compiled = $this->compiler->compile($template, ['first' => 'Jane']);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_multiple_php_tags_using_blade()
    {
        $template = <<<'EOD'
@php echo "Hello $first"; @endphp
@php echo "Hello $first";
@php echo "Hello $first"; @endphp
EOD;

        $compiled = $this->compiler->compile($template, ['first' => 'Jane']);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_csrf_token_using_blade()
    {
        $template = <<<'EOD'
@csrf
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_method_token_using_blade()
    {
        $template = <<<'EOD'
@method('PUT')
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_error_directive_using_blade()
    {
        $template = <<<'EOD'
@error('foo')
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_json_directive_using_blade()
    {
        $template = <<<'EOD'
@json($data)
EOD;

        $compiled = $this->compiler->compile($template, ['data' => ['foo' => 'bar']]);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_include_directive_using_blade()
    {
        $template = <<<'EOD'
@include('foo', ['bar' => 'baz'])
@includeIf('view.name', ['some' => 'data'])
@includeWhen($boolean, 'view.name', ['some' => 'data'])
@includeUnless($boolean, 'view.name', ['some' => 'data'])
@includeFirst(['custom.admin', 'admin'], ['some' => 'data'])
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_while_directive_using_blade()
    {
        $template = <<<'EOD'
@while(true)Foo
Bar
Baz@endwhile
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_auth_directive_using_blade()
    {
        $template = <<<'EOD'
@auth(true)Foo
Bar
Baz@endauth
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_guest_directive_using_blade()
    {
        $template = <<<'EOD'
@guest(true)Foo
Bar
Baz@endguest
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_each_directive_using_blade()
    {
        $template = <<<'EOD'
@each('view.name', [], 'job', 'view.empty')
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_stack_push_prepend_directive_using_blade()
    {
        $template = <<<'EOD'
@stack('scripts')
@push('scripts')
    <script src="/example.js"></script>
@endpush
@prepend('scripts')
    This will be first...
@endprepend
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_inject_directive_using_blade()
    {
        $template = <<<'EOD'
@inject('metrics', 'App\Services\MetricsService')
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_components_using_blade()
    {
        $template = <<<'EOD'
@component('mail::message')
Foo
@endcomponent
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_extend_a_layout_using_blade()
    {
        $template = <<<'EOD'
@extends('layouts.site')
@section('content')
Foo bar
@endsection
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_sections_using_blade()
    {
        $template = <<<'EOD'
@section('content')
Foo bar
@endsection
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }

    /**
     * @test
     */
    public function it_cannot_compile_yields_using_blade()
    {
        $template = <<<'EOD'
@yield('content')
@hasSection('content')
@section('content')
Foo bar
@endsection
@endif
@sectionMissing('content')
Bar baz
@endif
EOD;

        $compiled = $this->compiler->compile($template);

        $this->assertEquals("", $compiled);
    }
}
