<?php

namespace Ignite\Program\Activity\Offers\Demos;

use Ignite\Activity\Contracts\OfferForm;
use Ignite\Activity\Domain\Schema\Fields;
use Ignite\Activity\Models\Form\ActivityForm;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Fluent;
use Illuminate\Support\HtmlString;
use Kris\LaravelFormBuilder\Fields\FormField;

class GeneralSales extends ActivityForm implements OfferForm
{
    /**
     * Build the offer activity form.
     *
     * @return void
     */
    public function buildForm()
    {
        $this->fromSchema();

        $this->setFormOptions([
            'id' => 'general-sales-form',
            'files' => true,
        ]);
    }

    /**
     * The fields customized for this form.
     *
     * @param Fields $fields
     *
     * @return array
     */
    protected function fields(Fields $fields)
    {
        $this->addInvoiceDocumentFile($fields);
        $this->addPreferredLanguageRadios($fields);
        $this->addAgreeTermsCheckbox($fields);
        $this->addBusinessTypeDropdown($fields);
        $this->addProductsSoldCheckboxes($fields);

        return [
            $fields->get('company')->label('Company Name'),
            $fields->get('amount')->label('Sale Amount')->frontendType('number')->sortable(),
//            $fields->get('sale_id')->label('Sale ID')->frontendType('number'),
//            $fields->get('business_type')
//                ->attr('class', 'form-control select2')
//                ->choices([
//                    1 => 'Sole Proprietor',
//                    2 => 'Limited'
//                ])
//                ->emptyValue('-- Please choose --')
//                ->selected(function ($data) {
//                    return $data ?? 1;
//                }),
//            $fields->get('preferred_language'),
//            $fields->get('products_sold'),
//            $fields->get('invoice_document')
//                ->searchable(false)
//                ->sortable(false)
//                ->disk('public')
//                ->path("{$this->getType()->code}/{$this->getOffer()->code}")
//                ->storeAs(function (UploadedFile $file) {
//                    return $file->getClientOriginalName();
//                }),
//            $fields->get('agree_terms')
//                ->frontendType('toggle')
//                ->labelAttr('class', 'control-label')
//                ->value(1),
        ];
    }

    // TODO: Do we need this? Multiple lineitems perhaps?
    public function query(\Illuminate\Database\Eloquent\Builder $query)
    {
        //
    }

    public function formatters(\Illuminate\Support\Collection $fields)
    {
        return [
            'activity.data.business_type' => function ($model) use ($fields) {
                $value = $model->activity->data->business_type;
                $field = $fields->get('business_type');

                return $this->getValueFromChoices($field, $value);
            },
            'activity.data.preferred_language' => function ($model) use ($fields) {
                $value = $model->activity->data->preferred_language;
                $field = $fields->get('preferred_language');

                return $this->getValueFromChoices($field, $value);
            },
            'activity.data.products_sold' => function ($model) use ($fields) {
                $value = $model->activity->data->products_sold ?? [];
                $field = $fields->get('products_sold');

                return collect($field->getOption('choices'))
                    ->map(function ($name, $id) {
                        return new Fluent(compact('id', 'name'));
                    })
                    ->whereIn('id', $value)
                    ->pluck('name')
                    ->implode(', ');
            },
            'activity.data.invoice_document' => function ($model) use ($fields) {
                $field = $fields->get('invoice_document');
                $disk = $field->getOption('disk');
                $path = $field->getOption('path');
                $file = $model->activity->data->invoice_document;
                $filepath = "{$path}/{$file}";

                if ($file && Storage::disk($disk)->exists($filepath)) {
                    return new HtmlString(sprintf(
                        '<a href="%s" target="_blank">Download</a>',
                        Storage::disk($disk)->url($filepath)
                    ));
                }

                return '';
            },
            'activity.data.agree_terms' => function ($model) {
                return $model->activity->data->agree_terms === 1 ? 'Yes' : 'No';
            }
        ];
    }

    public function conditionals()
    {
        return [

        ];
    }

    private function addInvoiceDocumentFile(Fields $fields)
    {
        $fields->add([
            'name' => 'invoice_document',
            'label_alias' => 'Invoice',
            'label' => 'Invoice Attachment',
            'table' => 'program_sales',
            'frontend_type' => 'file',
            'backend_type' => 'string',
            'rules' => ['max:2048|mimes:pdf'],
            'attr' => ['class' => 'filestyle'],
        ]);
    }

    private function addPreferredLanguageRadios(Fields $fields)
    {
        $fields->add([
            'name' => 'preferred_language',
            'label_alias' => 'Language',
            'table' => 'program_sales',
            'frontend_type' => 'choice',
            'backend_type' => 'string',
            'rules' => ['required'],
            'attr' => ['class' => ''],
            'choices' => [
                'en' => 'English',
                'fr' => 'French',
                'es' => 'Spanish'
            ],
            'choice_options' => [
                'wrapper' => ['class' => 'choice-wrapper'],
                'label_attr' => ['class' => 'label-class'],
            ],
            'selected' => '2',
            'expanded' => true,
            'multiple' => false,
        ]);

        return $fields;
    }

    private function addBusinessTypeDropdown(Fields $fields)
    {
        $fields->add([
            'name' => 'business_type',
            'table' => 'program_sales',
            'frontend_type' => 'select',
            'backend_type' => 'string',
            'label' => 'Business Type',
            'rules' => ['required'],
            'empty_value' => '--Please choose--'
        ]);

        return $fields;
    }

    private function addProductsSoldCheckboxes(Fields $fields)
    {
        $fields->add([
            'name' => 'products_sold',
            'label_alias' => 'Products Sold',
            'label' => 'Products Sold',
            'table' => 'program_sales',
            'frontend_type' => 'choice',
            'backend_type' => 'array',
            'rules' => ['required'],
            'attr' => ['class' => ''],
            'choices' => [
                1 => 'A25-R',
                2 => 'A90-D',
                3 => 'C95-M'
            ],
            'choice_options' => [
                'wrapper' => ['class' => 'choice-wrapper'],
                'label_attr' => ['class' => 'label-class'],
            ],
            'selected' => function () {
                return [1, 3];
            },
            'expanded' => true,
            'multiple' => true,
        ]);

        return $fields;
    }

    private function addAgreeTermsCheckbox(Fields $fields)
    {
        $fields->add([
            'name' => 'agree_terms',
            'label_alias' => 'Agreed Terms',
            'table' => 'program_sales',
            'frontend_type' => 'checkbox',
            'backend_type' => 'boolean',
            'rules' => ['required'],
            'attr' => ['class' => ''],
            'single' => true,
        ]);

        return $fields;
    }

    private function getValueFromChoices(FormField $field, $value)
    {
        $choices = $field->getOption('choices', []);

        if (array_key_exists($value, $choices)) {
            return $choices[$value];
        }

        return $value;
    }
}
