<?php

namespace Ignite\Activity\Listeners;

use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Ignite\Core\Events\Menus\AdminMenuPreCompile;
use Ignite\Core\Models\Menu\Item;
use Nwidart\Modules\Facades\Module;

class AdminMenu
{
    /**
     * Handle the event.
     *
     * @param AdminMenuPreCompile $event
     * @return void
     */
    public function handle(AdminMenuPreCompile $event)
    {
        if (! Module::isEnabled('Activity')) {
            return;
        }

        $types = Type::query()->get()->map(function ($type, $index) {
            return $this->getItemFor($type, $index);
        });

        $item = [
            'url' => '#',
            'label' => 'Activities',
            'position' => 25,
            'allow' => auth()->user()->can("activity.browse"),
            'children' => (1 == $types->count()) ? $types[0]['children'] : $types->toArray(),
            'icon' => 'heartbeat',
        ];

        $event->getManager()->merge(
            $event->getManager()->hydrateItems(collect([$item]))
        );
    }

    /**
     * Get a menu item for a type.
     *
     * @param Type $type
     * @param int $index
     *
     * @return Item
     */
    protected function getItemFor($type, $index)
    {
        $position = ($index + 1) * 100;
        $offers = $this->getOffersFor($type);

        return [
            'position' => $position,
            'url' => route('admin.activity.type.index', ['type' => $type->code]),
            'label' => $type->plural(),
            'icon' => 'table',
            'allow' => $type->can('browse'),
            'children' => [
                [
                    'position' => $position + 1,
                    'url' => route('admin.activity.type.show', ['type' => $type->code]),
                    'label' => 'Submissions',
                    'allow' => $type->can('browse'),
                    'children' => (1 == count($offers)) ? $offers[0]['children'] : $offers,
                    'icon' => 'database',
                ],
                [
                    'position' => $position + 2,
                    'url' => url("/admin/activity/{$type->code}/offers"),
                    'label' => 'Offers',
                    'allow' => auth()->user()->can("activity.offer.browse"),
                    'children' => [],
                    'icon' => 'tag',
                ],
            ],
        ];
    }

    /**
     * @param Type $type
     * @return array
     */
    protected function getOffersFor($type)
    {
        $children = [];

        $offers = QueryPermissionFilters::for('claim.offer.offer.browse')->apply(
            $type->offers()->getQuery()
        )->get();

        /** @var Offer $offer */
        foreach ($offers as $offer) {
            $children[] = [
                'position' => $offer->sequence,
                'url' => route('admin.activity.index', ['type' => $type->code, 'offer' => $offer->code]),
                'label' => $offer->label(),
                'icon' => 'tag',
                'allow' => $type->can('browse'),
                'children' => [],
            ];
        }

        return $children;
    }
}
