<?php

namespace Ignite\Activity\Tests\Domain\Schema;

use Ignite\Activity\Domain\Schema\Field;
use Ignite\Activity\Tests\TestCase;
use InvalidArgumentException;

class FieldTest extends TestCase
{
    /**
     * @test
     */
    public function it_can_cast_to_array()
    {
        $field = new Field(['name' => 'foo']);

        $array = $field->toArray();

        $this->assertIsArray($array);
        $this->assertArrayHasKey('name', $array);
        $this->assertEquals('foo', $array['name']);
    }

    /**
     * @test
     */
    public function it_can_cast_to_json()
    {
        $field = new Field(['name' => 'foo']);

        $json = $field->toJson();

        $this->assertIsString($json);
        $this->assertStringContainsString('"name":"foo"', $json);
    }

    /**
     * @test
     */
    public function it_can_cast_to_json_with_options()
    {
        $field = new Field(['name' => 'foo']);

        $json = $field->toJson(JSON_PRETTY_PRINT);

        $this->assertIsString($json);
        $this->assertStringContainsString('"name": "foo"', $json);
    }

    /**
     * @test
     */
    public function it_can_be_serialized_to_json()
    {
        $field = new Field(['name' => 'foo']);

        $json = json_encode($field);

        $this->assertStringContainsString('"name":"foo"', $json);
    }

    /**
     * @test
     */
    public function it_can_set_a_single_wrapper_attr_property()
    {
        $field = new Field();

        $atts = ['class' => 'form-group', 'id' => 'foo-wrapper'];

        $field->wrapperAttr('id', 'foo-wrapper');
        $this->assertEquals($atts, $field->toArray()['wrapper']);
    }

    /**
     * @test
     */
    public function it_can_set_the_wrapper_attr_property()
    {
        $field = new Field();

        $field->wrapperAttrs(['class' => 'foo-group']);
        $this->assertEquals(['class' => 'foo-group'], $field->toArray()['wrapper']);
    }

    /**
     * @test
     */
    public function it_can_set_a_single_attr_property()
    {
        $field = new Field();

        $atts = ['class' => 'form-control', 'id' => 'foo-control'];

        $field->attr('id', 'foo-control');
        $this->assertEquals($atts, $field->toArray()['attr']);
    }

    /**
     * @test
     */
    public function it_can_set_the_attr_property()
    {
        $field = new Field();

        $field->attrs(['class' => 'foo-control']);
        $this->assertEquals(['class' => 'foo-control'], $field->toArray()['attr']);
    }

    /**
     * @test
     */
    public function it_can_set_the_help_block_tag_property()
    {
        $field = new Field();

        $field->helpBlockTag('div');

        $this->assertEquals(['tag' => 'div', 'text' => null, 'attr' => ['class' => 'help-block']], $field->toArray()['help_block']);
    }

    /**
     * @test
     */
    public function it_can_set_the_help_block_text_property()
    {
        $field = new Field();

        $field->helpBlockText('Help me');

        $this->assertEquals(['tag' => 'p', 'text' => 'Help me', 'attr' => ['class' => 'help-block']], $field->toArray()['help_block']);
    }

    /**
     * @test
     */
    public function it_can_set_a_single_help_block_attr_property()
    {
        $field = new Field();

        $atts = ['class' => 'help-block', 'id' => 'foo-help'];

        $field->helpBlockAttr('id', 'foo-help');
        $this->assertEquals($atts, $field->toArray()['help_block']['attr']);
    }

    /**
     * @test
     */
    public function it_can_set_the_help_block_attr_property()
    {
        $field = new Field();

        $field->helpBlockAttrs(['class' => 'foo-help']);
        $this->assertEquals(['class' => 'foo-help'], $field->toArray()['help_block']['attr']);
    }

    /**
     * @test
     */
    public function it_can_set_the_help_block_property()
    {
        $field = new Field();

        $field->helpBlock(['tag' => 'div', 'text' => 'Help me']);

        $this->assertEquals(['tag' => 'div', 'text' => 'Help me'], $field->toArray()['help_block']);
    }

    /**
     * @test
     */
    public function it_can_set_the_default_value_property()
    {
        $field = new Field();

        $field->defaultValue('foo');
        $this->assertEquals('foo', $field->toArray()['default_value']);
    }

    /**
     * @test
     */
    public function it_can_set_the_rules_property()
    {
        $field = new Field();

        $field->rules(['required']);
        $this->assertEquals(['required'], $field->toArray()['rules']);
    }

    /**
     * @test
     */
    public function it_can_set_the_errors_property()
    {
        $field = new Field();

        $field->errors(['data-foo' => 'bar']);
        $this->assertEquals(['data-foo' => 'bar'], $field->toArray()['errors']);
    }

    /**
     * @test
     */
    public function it_can_set_the_error_messages_property()
    {
        $field = new Field();

        $errorMessage = ['first.required' => 'The first name field is required.'];

        $field->errorMessages($errorMessage);

        $this->assertEquals($errorMessage, $field->toArray()['error_messages']);
    }

    /**
     * @test
     */
    public function it_cannot_set_the_name_property()
    {
        $this->expectException(InvalidArgumentException::class);

        $field = new Field();

        $field->name('Foo');
    }

    /**
     * @test
     */
    public function it_can_set_the_label_property()
    {
        $field = new Field();

        $field->label('Foo');
        $this->assertEquals('Foo', $field->toArray()['label']);
    }

    /**
     * @test
     */
    public function it_can_set_the_label_alias_property()
    {
        $field = new Field();

        $field->labelAlias('Foo Bar');
        $this->assertEquals('Foo Bar', $field->toArray()['label_alias']);
    }

    /**
     * @test
     */
    public function it_can_set_the_label_show_property()
    {
        $field = new Field();

        $field->labelShow();
        $this->assertTrue($field->toArray()['label_show']);

        $field->labelShow(false);
        $this->assertFalse($field->toArray()['label_show']);
    }

    /**
     * @test
     */
    public function it_can_set_a_single_label_attr_property()
    {
        $field = new Field();

        $field->labelAttr('id', 'foo-control');
        $this->assertEquals(['class' => 'control-label', 'id' => 'foo-control'], $field->toArray()['label_attr']);
    }

    /**
     * @test
     */
    public function it_can_set_the_label_attr_property()
    {
        $field = new Field();

        $field->labelAttrs(['class' => 'foo-control']);
        $this->assertEquals(['class' => 'foo-control'], $field->toArray()['label_attr']);
    }

    /**
     * @test
     */
    public function it_can_set_the_table_property()
    {
        $field = new Field();

        $field->table('foo_bar');
        $this->assertEquals('foo_bar', $field->toArray()['table']);
    }

    /**
     * @test
     */
    public function it_can_set_the_frontend_type_property()
    {
        $field = new Field();

        $field->frontendType('number');
        $this->assertEquals('number', $field->toArray()['frontend_type']);
    }

    /**
     * @test
     */
    public function it_can_set_the_backend_type_property()
    {
        $field = new Field();

        $field->backendType('int');
        $this->assertEquals('int', $field->toArray()['backend_type']);
    }

    /**
     * @test
     */
    public function it_can_set_the_internal_property()
    {
        $field = new Field();

        $field->internal();
        $this->assertTrue($field->toArray()['internal']);

        $field->internal(false);
        $this->assertFalse($field->toArray()['internal']);
    }

    /**
     * @test
     */
    public function it_can_set_the_orderable_property()
    {
        $field = new Field();

        $field->orderable();
        $this->assertTrue($field->toArray()['orderable']);

        $field->orderable(false);
        $this->assertFalse($field->toArray()['orderable']);
    }

    /**
     * @test
     */
    public function it_can_set_the_searchable_property()
    {
        $field = new Field();

        $field->searchable();
        $this->assertTrue($field->toArray()['searchable']);

        $field->searchable(false);
        $this->assertFalse($field->toArray()['searchable']);
    }

    /**
     * @test
     */
    public function it_can_set_the_exportable_property()
    {
        $field = new Field();

        $field->exportable();
        $this->assertTrue($field->toArray()['exportable']);

        $field->exportable(false);
        $this->assertFalse($field->toArray()['exportable']);
    }

    /**
     * @test
     */
    public function it_can_set_the_sortable_property()
    {
        $field = new Field();

        $field->sortable();
        $this->assertTrue($field->toArray()['sortable']);

        $field->sortable(false);
        $this->assertFalse($field->toArray()['sortable']);
    }

    /**
     * @test
     */
    public function it_can_set_the_visible_property()
    {
        $field = new Field();

        $field->visible();
        $this->assertTrue($field->toArray()['visible']);

        $field->visible(false);
        $this->assertFalse($field->toArray()['visible']);
    }

    /**
     * @test
     */
    public function it_can_dynamically_assign_keys_and_values_via_magic_call()
    {
        $field = new Field();

        $field->foo('bar')->barBaz('qux');

        $this->assertEquals('bar', $field->toArray()['foo']);
        $this->assertEquals('qux', $field->toArray()['bar_baz']);
    }

    /**
     * @test
     */
    public function it_can_will_not_allow_name_to_be_mutated_via_magic_call()
    {
        $this->expectException(InvalidArgumentException::class);

        $field = new Field();

        $field->__call('name', ['foo']);
    }

    /**
     * @test
     */
    public function it_can_support_checkbox_field_type()
    {
        $field = new Field();

        $field->value(1)->checked();

        $this->assertEquals(1, $field->toArray()['value']);
        $this->assertTrue($field->toArray()['checked']);
    }

    /**
     * @test
     */
    public function it_can_support_select_field_type()
    {
        $field = new Field();

        $choices = [
            1 => 'Active',
            2 => 'Inactive'
        ];

        $field->selected(1)->choices($choices)->emptyValue('Please choose');

        $this->assertEquals(1, $field->toArray()['selected']);
        $this->assertEquals($choices, $field->toArray()['choices']);
        $this->assertEquals('Please choose', $field->toArray()['empty_value']);
    }
}
