<?php

namespace Ignite\Activity\Domain\Schema;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use InvalidArgumentException;
use JsonSerializable;

/**
 * Field instance.
 *
 * @method self value(mixed $value)
 * @method self checked(bool $value = true)
 * @method self choices(array $value)
 * @method self choiceOptions(array $value)
 * @method self emptyValue(string $value)
 * @method self selected(mixed $value)
 * @method self expanded(bool $value = true)
 * @method self multiple(bool $value = true)
 * @method self tag(string $value)
 */
class Field implements Arrayable, Jsonable, JsonSerializable, \ArrayAccess
{
    /**
     * The possible schema data template.
     *
     * @var array
     */
    protected $data = [
        'wrapper' => ['class' => 'form-group'],
        'attr' => ['class' => 'form-control'],
        'help_block' => [
            'text' => null,
            'tag' => 'p',
            'attr' => ['class' => 'help-block'],
        ],
        'default_value' => null,
        'rules' => [],
        'errors' => ['class' => 'text-danger'],
        'error_messages' => [],
        'name' => '',
        'label' => '',
        'label_alias' => '',
        'label_show' => true,
        'label_attr' => ['class' => 'control-label'],
        'table' => '',
        'frontend_type' => 'text',
        'backend_type' => 'string',
        'internal' => false,
        'orderable' => true,
        'searchable' => true,
        'exportable' => true,
        'sortable' => true,
        'visible' => true,
    ];

    /**
     * Field constructor.
     *
     * @param array $data
     */
    public function __construct(array $data = [])
    {
        $this->data = array_replace_recursive($this->data, $data);
    }

    /**
     * Set a single wrapper attribute.
     *
     * @param string $key
     * @param string $value
     *
     * @return self
     */
    public function wrapperAttr(string $key, string $value)
    {
        Arr::set($this->data, "wrapper.$key", $value);

        return $this;
    }

    /**
     * Set the attributes.
     *
     * @param mixed $value
     *
     * @return self
     */
    public function wrapperAttrs(array $value)
    {
        Arr::set($this->data, 'wrapper', $value);

        return $this;
    }

    /**
     * Set a single attribute.
     *
     * @param string $key
     * @param string $value
     *
     * @return self
     */
    public function attr(string $key, string $value)
    {
        Arr::set($this->data, "attr.$key", $value);

        return $this;
    }

    /**
     * Set the wrapper attributes.
     *
     * @param mixed $value
     *
     * @return self
     */
    public function attrs(array $value)
    {
        Arr::set($this->data, 'attr', $value);

        return $this;
    }

    /**
     * Set the help block tag.
     *
     * @param string $value
     *
     * @return self
     */
    public function helpBlockTag(string $value)
    {
        Arr::set($this->data, 'help_block.tag', $value);

        return $this;
    }

    /**
     * Set the help block text.
     *
     * @param string $value
     *
     * @return self
     */
    public function helpBlockText(string $value)
    {
        Arr::set($this->data, 'help_block.text', $value);

        return $this;
    }

    /**
     * Set a single help block attribute.
     *
     * @param string $value
     *
     * @return self
     */
    public function helpBlockAttr(string $key, string $value)
    {
        Arr::set($this->data, "help_block.attr.$key", $value);

        return $this;
    }

    /**
     * Set a single help block attribute.
     *
     * @param array $value
     *
     * @return self
     */
    public function helpBlockAttrs(array $value)
    {
        Arr::set($this->data, "help_block.attr", $value);

        return $this;
    }

    /**
     * Set the help block property.
     *
     * @param mixed $value
     *
     * @return self
     */
    public function helpBlock(array $value)
    {
        Arr::set($this->data, 'help_block', $value);

        return $this;
    }

    /**
     * Set the default value.
     *
     * @param mixed $value
     *
     * @return self
     */
    public function defaultValue($value)
    {
        Arr::set($this->data, 'default_value', $value);

        return $this;
    }

    /**
     * Set the field validation rules.
     *
     * @param array $value
     *
     * @return self
     */
    public function rules(array $value)
    {
        Arr::set($this->data, 'rules', $value);

        return $this;
    }

    /**
     * Set the field errors.
     *
     * @param array $value
     *
     * @return self
     */
    public function errors(array $value)
    {
        Arr::set($this->data, 'errors', $value);

        return $this;
    }

    /**
     * Set the field error messages.
     *
     * @param array $value
     *
     * @return self
     */
    public function errorMessages(array $value)
    {
        Arr::set($this->data, 'error_messages', $value);

        return $this;
    }

    /**
     * Do not allow the name to change.
     *
     * @param string $value
     *
     * @throws InvalidArgumentException
     */
    public function name(string $value)
    {
        $this->guardAgainstNameMutation();
    } // @codeCoverageIgnore

    /**
     * Set the field label.
     *
     * @param string $value
     *
     * @return self
     */
    public function label(string $value)
    {
        Arr::set($this->data, 'label', $value);

        return $this;
    }

    /**
     * Set the field label alias.
     *
     * @param string $value
     *
     * @return self
     */
    public function labelAlias(string $value)
    {
        Arr::set($this->data, 'label_alias', $value);

        return $this;
    }

    /**
     * Set the field to show or hide the label.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function labelShow(bool $flag = true)
    {
        Arr::set($this->data, 'label_show', $flag);

        return $this;
    }

    /**
     * Set a single label attribute.
     *
     * @param string $key
     * @param string $value
     *
     * @return self
     */
    public function labelAttr(string $key, string $value)
    {
        Arr::set($this->data, "label_attr.$key", $value);

        return $this;
    }

    /**
     * Set the field label attributes.
     *
     * @param array $value
     *
     * @return self
     */
    public function labelAttrs(array $value)
    {
        Arr::set($this->data, "label_attr", $value);

        return $this;
    }

    /**
     * Set the field table.
     *
     * @param string $value
     *
     * @return self
     */
    public function table(string $value)
    {
        Arr::set($this->data, 'table', $value);

        return $this;
    }

    /**
     * Set the field frontend type.
     *
     * @param string $value
     *
     * @return self
     */
    public function frontendType(string $value)
    {
        Arr::set($this->data, 'frontend_type', $value);

        return $this;
    }

    /**
     * Set the field backend type.
     *
     * @param string $value
     *
     * @return self
     */
    public function backendType(string $value)
    {
        Arr::set($this->data, 'backend_type', $value);

        return $this;
    }

    /**
     * Set the field as internal.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function internal(bool $flag = true)
    {
        Arr::set($this->data, 'internal', $flag);

        return $this;
    }

    /**
     * Set the field as orderable.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function orderable(bool $flag = true)
    {
        Arr::set($this->data, 'orderable', $flag);

        return $this;
    }

    /**
     * Set the field as searchable.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function searchable(bool $flag = true)
    {
        Arr::set($this->data, 'searchable', $flag);

        return $this;
    }

    /**
     * Set the field as exportable.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function exportable(bool $flag = true)
    {
        Arr::set($this->data, 'exportable', $flag);

        return $this;
    }

    /**
     * Set the field as sortable.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function sortable(bool $flag = true)
    {
        Arr::set($this->data, 'sortable', $flag);

        return $this;
    }

    /**
     * Set the field as visible.
     *
     * @param bool $flag
     *
     * @return self
     */
    public function visible(bool $flag = true)
    {
        Arr::set($this->data, 'visible', $flag);

        return $this;
    }

    /**
     * Cast the field to an array.
     *
     * @return array
     */
    public function toArray()
    {
        return $this->data;
    }

    /**
     * Cast the field to JSON.
     *
     * @param int $options
     * @return false|string
     */
    public function toJson($options = 0)
    {
        return json_encode($this->jsonSerialize(), $options);
    }

    /**
     * Serialize the field as json.
     *
     * @return array
     */
    public function jsonSerialize(): mixed
    {
        return $this->toArray();
    }

    /**
     * Magic call method will convert the following methods into key -> value pairs.
     *
     * $field->foo() => 'foo' => true
     * $field->foo(false) => 'foo' => false
     * $field->foo('bar') => 'foo' => 'bar'
     * $field->fooBar('baz') => 'foo_bar' => 'baz'
     *
     * @param string $name
     * @param array $arguments
     *
     * @return self
     */
    public function __call(string $name, array $arguments)
    {
        if ($name === 'name') {
            $this->guardAgainstNameMutation();
        }

        $key = strtolower(Str::snake($name));
        $value = empty($arguments) ? true : array_shift($arguments);

        Arr::set($this->data, $key, $value);

        return $this;
    }

    /**
     * The exception to throw when someone attempts to mutate the name property.
     *
     * @throws InvalidArgumentException
     */
    protected function guardAgainstNameMutation()
    {
        throw new InvalidArgumentException('You cannot change the name. Use remove($name) and add($field) instead.');
    }

    /**
     * Whether a offset exists
     * @link https://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset
     * @return bool true on success or false on failure.
     */
    public function offsetExists(mixed $offset): bool
    {
        return isset($this->data[$offset]);
    }

    /**
     * Offset to retrieve
     * @link https://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset
     * @return mixed Can return all value types.
     */
    public function offsetGet(mixed $offset): mixed
    {
        return $this->data[$offset];
    }

    /**
     * Offset to set
     * @link https://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset
     * @param mixed $value
     * @return void
     */
    public function offsetSet(mixed $offset, mixed $value): void
    {
        throw new \LogicException("Cannot set $offset. Use class methods.");
    }

    /**
     * Offset to unset
     * @link https://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset
     * @return void
     */
    public function offsetUnset(mixed $offset): void
    {
        throw new \LogicException("Cannot unset $offset.");
    }
}
