<?php

namespace Ignite\Activity\Domain\Offers;

use Exception;
use Ignite\Activity\Contracts\OfferForm as OfferFormContract;
use Ignite\Activity\Entities\Offer;
use Kris\LaravelFormBuilder\FormBuilder;

class OfferFormFactory
{
    /**
     * The registered Offer classes.
     *
     * @var array
     */
    protected $forms = [];

    /**
     * @var FormBuilder
     */
    private $formBuilder;

    /**
     * OfferFormFactory instance.
     *
     * @param FormBuilder $formBuilder
     * @param array $forms
     */
    public function __construct(FormBuilder $formBuilder, array $forms = [])
    {
        $this->forms = $forms;
        $this->formBuilder = $formBuilder;
    }

    /**
     * Register an OfferForm class using a short-name as the key.
     *
     * @param string $name
     * @param string $class
     *
     * @return $this
     */
    public function register($name, $class)
    {
        $this->forms[$name] = $class;

        return $this;
    }

    /**
     * Deregister an OfferForm class using a short-name as the key.
     *
     * @param string $name
     *
     * @return $this
     */
    public function deregister($name)
    {
        unset($this->forms[$name]);

        return $this;
    }

    /**
     * Check if the given name and/or class has already been registered.
     *
     * @param string $name
     * @param string|null $class
     * @return false
     */
    public function has(string $name, ?string $class = null)
    {
        if (array_key_exists($name, $this->forms) && is_null($class)) {
            return true;
        }

        if (array_key_exists($name, $this->forms) && $class === $this->forms[$name]) {
            return true;
        }

        return false;
    }

    /**
     * Make an instance of an Offer class given the short-name, options and data.
     *
     * @param string $name
     * @param array $options
     * @param array $data
     *
     * @return OfferFormContract
     * @throws Exception
     */
    public function make($name, array $options = [], array $data = [])
    {
        if (! $this->has($name)) {
            //throw OfferException::notFound($name);
            throw new Exception("{$name} not registered.");
        }

        if (! isset($data['offer']) || ! $data['offer'] instanceof Offer) {
            //throw OfferException::missingModel($name, $data);
            throw new Exception("The offer was not passed in with the data parameters.");
        }

        $instance = $this->formBuilder->create($this->forms[$name], $options, $data);

        if (! $instance instanceof OfferFormContract) {
            //throw OfferException::missingInterface($instance, $name);
            throw new Exception('Form must implement OfferFormContract');
        }

        return $instance;
    }
}
