<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Domain\Rules\RuleFactory;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Rule;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Carbon;
use Mockery;

class RuleTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_provides_a_method_to_get_the_label()
    {
        /** @var Rule $rule */
        $rule = factory(Rule::class)->create([
            'related_name' => 'MyFirstRule',
        ]);

        $this->assertEquals('MyFirstRule', $rule->relatedName());
    }

    /**
     * @test
     */
    public function it_casts_meta_from_json_to_array()
    {
        $rule = factory(Rule::class)->create([
            'meta' => [
                'foo' => 'bar'
            ]
        ]);

        $this->assertEquals('bar', $rule->meta['foo']);
    }

    /**
     * @test
     */
    public function it_can_get_the_related_rule_logic_instance()
    {
        /** @var RuleFactory $ruleFactory */
        $ruleFactory = app(RuleFactory::class);

        $ruleClass = Mockery::mock(RuleContract::class);

        $ruleFactory->register('WithinDateBounds', get_class($ruleClass));

        /** @var Rule $rule */
        $rule = factory(Rule::class)->create([
            'related_name' => 'WithinDateBounds'
        ]);

        /** @var Offer $offer */
        $offer = factory(Offer::class)->create();

        $offer->rules()->save($rule);

        $this->assertInstanceOf(RuleContract::class, $rule->instance());
    }

    /**
     * @test
     */
    public function it_provides_an_easy_way_to_get_meta()
    {
        /** @var Rule $rule */
        $rule = factory(Rule::class)->create([
            'meta' => [
                'available_at' => $start = Carbon::now()->startOfMonth()->toDateTimeString()
            ]
        ]);

        $this->assertEquals($start, $rule->meta('available_at'));
    }

    /**
     * @test
     */
    public function it_provides_an_easy_way_to_set_meta()
    {
        /** @var Rule $rule */
        $rule = factory(Rule::class)->create([
            'meta' => [
                'foo' => [
                    'bar' => 'baz'
                ]
            ]
        ]);

        $rule->metaSet('foo.bar', 'changed');

        $this->assertEquals('changed', $rule->meta('foo.bar'));
    }

    /**
     * @test
     */
    public function it_provides_an_easy_way_to_save_meta()
    {
        /** @var Rule $rule */
        $rule = factory(Rule::class)->create([
            'meta' => [
                'foo' => [
                    'bar' => 'baz'
                ]
            ]
        ]);

        $rule->metaSave('foo.bar', 'changed');

        $rule = Rule::find($rule->getKey());

        $this->assertEquals('changed', $rule->meta('foo.bar'));
    }

    /**
     * @test
     */
    public function it_defines_a_many_to_many_relationship_to_the_offers()
    {
        /** @var Rule $rule */
        $rule = factory(Rule::class)->create([
            'related_name' => 'MaxParticipants',
            'meta' => [
                'total' => 50
            ]
        ]);

        /** @var Offer $offers */
        $offers = factory(Offer::class, 2)->make();

        foreach ($offers as $offer) {
            $rule->offers()->save($offer);
        }

        $this->assertCount(2, $rule->offers);
    }
}
